package com.rafian.random.randomX;

/**
 * <p>
 * Implementation of a <b>randomX</b>-compliant class using the "Minimal Standard" multiplicative congruential generator
 * of Park and Miller. [Park, S.K. and K.W. Miller, <cite>Communications of the ACM</cite> <b>31</b>, 1192-1201 (1988).]
 * </p>
 *
 * <p>
 * The generation algorithm is: <center> <em>I<sub>j+1</sub></em> = ( <em>I<sub>j</sub></em>  16807) & 0x7FFFFFFF
 * </center>
 * </p>
 *
 * <p>
 * Note that the intermediate value of the multiplication by 16807 (7<sup>5</sup>) exceeds that representable in 32
 * bits; this has deterred use of this generator in most portable languages. Fortunately, Java's <tt>long</tt> type is
 * guaranteed to be 64 bits, so a straightforward and portable implementation is possible.
 * </p>
 *
 * <ul>
 * <li>First Designed and implemented in July 1996 by <a href="http://www.fourmilab.ch/">John Walker</a>.</li>
 * <li>Updated 2014 by Ferret Renaud.</li>
 * </ul>
 */
public class RandomMCG extends AbstractRandomX {
  private static final long serialVersionUID = 1L;
  private long state;

  /**
   * Constructor.
   *
   * @throws IllegalArgumentException
   *           never here
   */
  public RandomMCG() throws IllegalArgumentException {
    this(System.currentTimeMillis());
  }

  /**
   * Constructor.
   *
   * @param seed
   *          initial seed for the generator
   * @throws IllegalArgumentException
   *           if seed is invalid
   */
  public RandomMCG(long seed) throws IllegalArgumentException {
    super();
    this.setSeed(seed);
  }

  /**
   * Set seed for generator. Subsequent values will be based on the given nonzero seed.
   *
   * @param seed
   *          seed for the generator
   * @throws IllegalArgumentException
   *           if seed is invalid
   */
  @Override
  public void setSeed(long seed) throws IllegalArgumentException {
    if (seed == 0) {
      throw new IllegalArgumentException("seed must be nonzero");
    }
    super.resetSeed(); // Notify parent seed has changed
    this.state = seed & 0xFFFFFFFFL;
    for (int i = 0; i < 11; i++) {
      this.nextByte();
    }
  }

  @Override
  public byte nextByte() {
    this.state = (this.state * 16807) & 0x7FFFFFFFL;
    return (byte) ((this.state >> 11) & 0xFF);
  }
}
