package com.rafian.random.randomX;

/**
 * <p>
 * Implementation of a <b>randomX</b>-compliant class using L'Ecuyer's two-sequence generator with a Bays-Durham
 * shuffle, as described on page 282 of Press et al., <cite>Numerical Recipes in C</cite>, 2nd edition. Their
 * implementation was constrained by the absence of a 64-bit integer data type. Since Java guarantees a <tt>long</tt> to
 * be 64 bit, we can use L'Ecuyer's multiplier and modulus directly, rather than failing around with Schrage's
 * algorithm. Further, 64-bit <tt>long</tt> arithmetic allows us to directly combine the results from the two generators
 * by adding and taking the modulus of one of them, bypassing the subtract and test for negative gimmick used in
 * <cite>Numerical Recipes</cite>.
 * </p>
 *
 * <p>
 * For additional details, see L'Ecuyer's original 1968 paper at page 742 of <cite>Communications of the ACM</cite>,
 * Vol. 31.
 * </p>
 *
 * <ul>
 * <li>First Designed and implemented in July 1996 by <a href="http://www.fourmilab.ch/">John Walker</a>.</li>
 * <li>Updated 2014 by Ferret Renaud.</li>
 * </ul>
 */

public class RandomLEcuyer extends AbstractRandomX {
  private static final long serialVersionUID = 1L;
  /*
   * L'Ecuyer's recommended multiplier and modulus for the two multiplicative congruential generators. Even though the
   * values fit in 32 bits, we declare them as long so that the arithmetic in calculating the next value will be
   * automatically done in long without need for casting.
   */
  private static final long mul1 = 40014;
  private static final long mod1 = 2147483563;
  private static final long mul2 = 40692;
  private static final long mod2 = 2147483399;

  // Shuffle table size
  private static final int shuffleSize = 32;
  // Number of initial warm up results to "burn"
  private static final int warmup = 19;

  private int gen1, gen2, state;
  private int[] shuffle;

  /**
   * Constructor.
   *
   * @throws IllegalArgumentException
   *           never here
   */
  public RandomLEcuyer() throws IllegalArgumentException {
    this(System.currentTimeMillis());
  }

  /**
   * Constructor.
   *
   * @param seed
   *          initial seed for the generator
   * @throws IllegalArgumentException
   *           if seed is invalid
   */
  public RandomLEcuyer(long seed) throws IllegalArgumentException {
    super();
    this.shuffle = new int[RandomLEcuyer.shuffleSize];
    this.setSeed(seed);
  }

  /**
   * Set seed for generator. Subsequent values will be based on the given nonzero seed.
   *
   * @param seed
   *          seed for the generator
   * @throws IllegalArgumentException
   *           if seed is invalid
   */
  @Override
  public void setSeed(long seed) throws IllegalArgumentException {
    super.resetSeed();
    if (seed == 0) {
      throw new IllegalArgumentException("seed must be nonzero");
    }
    this.gen1 = this.gen2 = (int) (seed & 0x7FFFFFFFL);

    // "Warm up" the generator for a number of rounds to eliminate any
    // residual influence of the seed.
    for (int i = 0; i < RandomLEcuyer.warmup; i++) {
      this.gen1 = (int) ((this.gen1 * RandomLEcuyer.mul1) % RandomLEcuyer.mod1);
    }

    // Fill the shuffle table with values
    for (int i = 0; i < RandomLEcuyer.shuffleSize; i++) {
      this.gen1 = (int) ((this.gen1 * RandomLEcuyer.mul1) % RandomLEcuyer.mod1);
      this.shuffle[(RandomLEcuyer.shuffleSize - 1) - i] = this.gen1;
    }
    this.state = this.shuffle[0];
  }

  @Override
  public byte nextByte() {
    // Cycle generator 1
    this.gen1 = (int) ((this.gen1 * RandomLEcuyer.mul1) % RandomLEcuyer.mod1);
    // Cycle generator 2
    this.gen2 = (int) ((this.gen2 * RandomLEcuyer.mul2) % RandomLEcuyer.mod2);

    // Extract shuffle table index from most significant part of the
    // previous result.
    int i = this.state / (1 + ((((int) RandomLEcuyer.mod1) - 1) / RandomLEcuyer.shuffleSize));

    // New state is sum of generators modulo one of their moduli
    this.state = (int) ((((long) this.shuffle[i]) + this.gen2) % RandomLEcuyer.mod1);

    // Replace value in shuffle table with generator 1 result
    this.shuffle[i] = this.gen1;
    return (byte) (this.state / (1 + ((((int) RandomLEcuyer.mod1) - 1) / 256)));
  }
}
