package com.rafian.random.randomX;

/**
 * Implementation of a <b>randomX</b>-compliant class using the simple (and not very good) <tt>rand()</tt> linear
 * congruential generator given as an example in the ANSI C specification. This is intended not for serious use, merely
 * as an illustration of a simple software-based <b>randomX</b> generator.
 *
 * <p>
 * The generation algorithm is:
 *
 * <p>
 * <center> <em>I<sub>j+1</sub></em> = (<em>I<sub>j</sub></em>  1103515245 + 12345) & 0x7FFFFFFF </center>
 *
 * <ul>
 * <li>First Designed and implemented in July 1996 by <a href="http://www.fourmilab.ch/">John Walker</a>.</li>
 * <li>Updated 2014 by Ferret Renaud.</li>
 * </ul>
 */

public class RandomLCG extends AbstractRandomX {
  private static final long serialVersionUID = 1L;
  private long state;

  /**
   * Constructor.
   */
  public RandomLCG() {
    this(System.currentTimeMillis());
  }

  /**
   * Creates a new pseudorandom number generator with a specified seed.
   *
   * @param seed
   *          initial seed for the generator
   */
  public RandomLCG(long seed) {
    super();
    this.setSeed(seed);
  }

  /**
   * Set seed for generator. Subsequent values will be based on the given seed.
   *
   * @param seed
   *          seed for the generator
   */
  @Override
  public void setSeed(long seed) {
    super.resetSeed(); // Notify parent seed has changed
    this.state = seed & 0xFFFFFFFFL;
  }

  /**
   * Get next byte from generator. Given how poor this generator is, it's wise to make a separate call for each byte
   * rather than extract fields from a single call, which may be correlated. Also, the higher-order bits of this
   * generator are more random than the low, so we extract the byte after discarding the low-order 11 bits.
   *
   * @return the next byte from the generator.
   */
  @Override
  public byte nextByte() {
    this.state = ((this.state * 1103515245L) + 12345L) & 0x7FFFFFFFL;
    return (byte) ((this.state >> 11) & 0xFF);
  }
}
