package com.rafian.random.randomX;

import com.rafian.random.AbstractRandom;

/**
 * <p>
 * Abstract superclass for emulations of java.util.Random with various underlying generators. These generators provide a
 * superset of the methods of the built-in Java generator, and allow easy replacement of the low-level byte-stream
 * random generator without the need to re-implement the higher-level calls.
 * </p>
 * <p>
 * The nature of the data returned by the functions in this class depends upon the generator provided by the class
 * derived from it. If the generator is algorithmic, the data are pseudorandom; if a hardware generator is employed,
 * genuine random data may be obtained. For brevity, in this document, we use <em>random</em> to refer to the data
 * returned, whatever its actual source.
 * </p>
 *
 * <ul>
 * <li>First Designed and implemented in July 1996 by <a href="http://www.fourmilab.ch/">John Walker</a>.</li>
 * <li>Updated 2014 by Ferret Renaud.</li>
 * </ul>
 */
public abstract class AbstractRandomX extends AbstractRandom {
  private static final long serialVersionUID = 1L;
  private double gset;
  private int nbits = 0;
  private boolean iset = false;
  private byte b;

  /**
   * Constructor.
   */
  public AbstractRandomX() {
    super();
  }

  @Override
  public boolean[] nextBooleans(int aNumber) {
    boolean[] resu = new boolean[aNumber];
    for (int i = 0; i < aNumber; i++) {
      resu[i] = this.nextBoolean();
    }
    return resu;
  }

  @Override
  public int[] nextInts(int aNumber, int aMin, int aMax) {
    int[] resu = new int[aNumber];
    for (int i = 0; i < aNumber; i++) {
      resu[i] = this.nextInt(aMin, aMax);
    }
    return resu;
  }

  /**
   * Reset when seed changes. A generator which supports seed must call this method by <tt>super.setSeed()</tt> when its
   * own <tt>setSeed(</tt> <i>long</i><tt>)</tt> method is called. This allows randomX to discard any buffered data in
   * the <tt>nextBit()</tt> and <tt>nextGaussian()</tt> methods so that subsequent calls will immediately reflect the
   * new seed.
   *
   * <p>
   * If a derived class does not permit specification of a seed (hardware-based generators, for example), it should
   * declare:
   *
   * <p>
   * <blockquote> <tt>private void setSeed(long seed) { }</tt> </blockquote>
   * <p>
   * which will hide the setSeed method from its users and cause a compile-time error if a program attempts to specify a
   * seed.
   */
  protected void resetSeed() {
    this.nbits = 0;
    this.iset = false;
  }

  @Override
  public int nextInt() {
    return ((this.nextShort()) << 16) | ((this.nextShort()) & 0xFFFF);
  }

  @Override
  public int nextInt(int aMin, int aMax) {
    int val = this.nextInt();
    if (aMin >= 0) {
      val = Math.abs(val) % aMax;
    } else {
      val = val % aMax;
    }
    if (val < aMin) {
      val = aMin;
    }
    return val;
  }

  @Override
  public long nextLong() {
    return (((long) this.nextInt()) << 32) | ((this.nextInt()) & 0xFFFFFFFFl);
  }

  @Override
  public float nextFloat() {
    return (float) ((this.nextInt() & 0x7FFFFFFF) / (0x7FFFFFFF * 1.0));
  }

  @Override
  public double nextDouble() {
    return (this.nextLong() & 0x7FFFFFFFFFFFFFFFl) / (0x7FFFFFFFFFFFFFFFl * 1.0);
  }

  @Override
  public double nextGaussian() {
    double fac, rsq, v1, v2;

    if (!this.iset) {
      do {
        v1 = (2 * this.nextDouble()) - 1;
        v2 = (2 * this.nextDouble()) - 1;
        rsq = (v1 * v1) + (v2 * v2);
      } while ((rsq > 1.0) || (rsq == 0.0));
      fac = Math.sqrt((-2.0 * Math.log(rsq)) / rsq);
      this.gset = v1 * fac;
      this.iset = true;
      return v2 * fac;
    }
    this.iset = false;
    return this.gset;
  }

  // Extended generator access methods with default implementations

  @Override
  public boolean nextBoolean() {
    boolean bit;

    if (this.nbits <= 0) {
      this.b = this.nextByte();
      this.nbits = 8;
    }
    bit = (this.b & 0x80) != 0;
    this.b <<= 1;
    this.nbits--;
    return bit;
  }

  /**
   * Fill a portion of an array of bytes with random data.
   *
   * @param buf
   *          array of <tt>byte</tt> to fill.
   * @param buflen
   *          number of bytes to store.
   */

  protected void nextByte(byte buf[], int buflen) {
    int i = 0;
    while (buflen-- > 0) {
      buf[i++] = this.nextByte();
    }
  }

  /**
   * Fill an array of bytes with random data.
   *
   * @param buf
   *          array of <tt>byte</tt>s to fill.
   */
  public void nextByte(byte buf[]) {
    this.nextByte(buf, buf.length);
  }

  @Override
  public short nextShort() {
    return (short) (((this.nextByte()) << 8) | ((short) (this.nextByte() & 0xFF)));
  }
}
