package com.rafian.random;

import java.util.Random;

/**
 * Implementation of a <b>randomX</b>-compliant class based upon the built-in <tt>Java.util.Random</tt> generator. Note
 * that since the higher level result composition methods are different from those in the (undocumented) standard
 * library, <b>randomJava</b> results will differ from those of the standard library for a given seed.
 *
 * <ul>
 * <li>First Designed and implemented in July 1996 by <a href="http://www.fourmilab.ch/">John Walker</a>.</li>
 * <li>Updated 2014 by Ferret Renaud.</li>
 * </ul>
 */
public class RandomJava extends AbstractRandom {
  private static final long serialVersionUID = 1L;
  private Random r;

  /**
   * Constructor.
   */
  public RandomJava() {
    super();
    this.r = new Random();
  }

  /**
   * Constructor.
   *
   * @param seed
   *          initial seed for the generator
   */
  public RandomJava(long seed) {
    super();
    this.r = new Random(seed);
  }

  /**
   * Set seed for generator. Subsequent values will be based on the given seed.
   *
   * @param seed
   *          seed for the generator
   */
  @Override
  public void setSeed(long seed) {
    this.r.setSeed(seed);
  }

  @Override
  public boolean nextBoolean() {
    return this.r.nextBoolean();
  }

  @Override
  public int[] nextInts(int aNumber, int aMin, int aMax) {
    int[] resu = new int[aNumber];
    for (int i = 0; i < aNumber; i++) {
      resu[i] = this.nextInt(aMin, aMax);
    }
    return resu;
  }

  @Override
  public boolean[] nextBooleans(int aNumber) {
    boolean[] resu = new boolean[aNumber];
    for (int i = 0; i < aNumber; i++) {
      resu[i] = this.r.nextBoolean();
    }
    return resu;
  }

  /**
   * Move from a byte from the array
   *
   * @param buf
   *          an array of byte
   */
  public void nextBytes(byte[] buf) {
    this.r.nextBytes(buf);
  }

  @Override
  public double nextDouble() {
    return this.r.nextDouble();
  }

  @Override
  public float nextFloat() {
    return this.r.nextFloat();
  }

  @Override
  public double nextGaussian() {
    return this.r.nextGaussian();
  }

  @Override
  public int nextInt() {
    return this.r.nextInt();
  }

  @Override
  public int nextInt(int aMin, int aMax) {
    int val = this.r.nextInt(aMax);
    if (val < aMin) {
      val = aMin;
    }
    return val;
  }

  @Override
  public long nextLong() {
    return this.r.nextLong();
  }

  @Override
  public int hashCode() {
    return this.r.hashCode();
  }

  @Override
  public byte nextByte() {
    byte[] d = new byte[1];
    this.r.nextBytes(d);
    return d[0];
  }

}
