package com.rafian;

import java.io.IOException;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.Level;
import org.apache.log4j.Priority;

import com.rafian.random.IRandom;
import com.rafian.random.RandomJava;

/**
 * Controller.
 */
public class RafianController extends HttpServlet {

  private static final long serialVersionUID = 1L;

  /** Default class to use. */
  private final static Class<RandomJava> DEFAULT_RANDOM = com.rafian.random.RandomJava.class;

  @SuppressWarnings("rawtypes")
  private Class randomClass;
  private long randomSeed;

  /**
   * Constructor of the object.
   */
  public RafianController() {
    super();
  }

  @Override
  public void init(ServletConfig aConfig) throws ServletException {
    super.init(aConfig);
    String randomClassName = aConfig.getInitParameter("randomclassname");
    try {
      this.randomClass = Class.forName(randomClassName);
    } catch (ClassNotFoundException exp) {
      this.logError("Cannot find object '", randomClassName, "', will use default class ",
          RafianController.DEFAULT_RANDOM, exp);
      this.randomClass = RafianController.DEFAULT_RANDOM;
    }

    String randomSeedParam = aConfig.getInitParameter("randomseed");
    if ((randomSeedParam != null) && !randomSeedParam.trim().isEmpty() && !"auto".equalsIgnoreCase(randomSeedParam)) {
      try {
        this.randomSeed = Long.parseLong(randomSeedParam);
      } catch (Throwable exp) {
        this.logError("Randomseed is not a valid number '", randomSeedParam, "', will use auto.",
            RafianController.DEFAULT_RANDOM, exp);
      }
    } else {
      this.randomSeed = System.currentTimeMillis();
    }

  }

  @Override
  protected void service(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
    HttpSession session = request.getSession();
    IRandom random = (IRandom) session.getAttribute("random");
    if (random == null) {
      try {
        random = (IRandom) this.randomClass.newInstance();
      } catch (Throwable exp) {
        this.logFatal("Cannot create object '", this.randomClass, "', will do nothing ... ", exp);
        return;
      }
      random.setSeed(this.randomSeed);
      session.setAttribute("random", random);
    }
    StringBuilder result = new StringBuilder();

    // This is inspired from 'old' api of random.org
    String numParam = request.getParameter("num");
    String minParam = request.getParameter("min");
    String maxParam = request.getParameter("max");
    // We do not user those
    // String colParam = request.getParameter("col");
    // String baseParam = request.getParameter("base");
    // String formatParam = request.getParameter("format");
    // String rndParam = request.getParameter("rnd");
    int num = Integer.parseInt(numParam);
    long min = Long.parseLong(minParam);
    long max = Long.parseLong(maxParam);
    if ((min == 0) && (max == 1)) {
      this.logDebug("Will generate boolean");
      for (int lcI = 0; lcI < num; lcI++) {
        result.append(random.nextBoolean() ? '1' : '0');
        if ((lcI + 1) <= num) {
          result.append('\n');
        }
      }
    } else if ((min == Short.MIN_VALUE) && (max == Short.MAX_VALUE)) {
      this.logDebug("Will generate short");
      for (int lcI = 0; lcI < num; lcI++) {
        result.append(random.nextShort());
        if ((lcI + 1) <= num) {
          result.append('\n');
        }
      }
    } else if ((min == Integer.MIN_VALUE) && (max == Integer.MAX_VALUE)) {
      this.logDebug("Will generate integer");
      for (int lcI = 0; lcI < num; lcI++) {
        result.append(random.nextInt());
        if ((lcI + 1) <= num) {
          result.append('\n');
        }
      }
    } else if ((min == Long.MIN_VALUE) && (max == Long.MAX_VALUE)) {
      this.logDebug("Will generate long");
      for (int lcI = 0; lcI < num; lcI++) {
        result.append(random.nextLong());
        if ((lcI + 1) <= num) {
          result.append('\n');
        }
      }
    } else if ((min == Byte.MIN_VALUE) && (max == Byte.MAX_VALUE)) {
      this.logDebug("Will generate byte");
      for (int lcI = 0; lcI < num; lcI++) {
        result.append(random.nextByte());
        if ((lcI + 1) <= num) {
          result.append('\n');
        }
      }
    } else {
      this.logDebug("Will generate integer with min max");
      for (int lcI = 0; lcI < num; lcI++) {
        result.append(random.nextInt((int) min, (int) max + 1));
        if ((lcI + 1) <= num) {
          result.append('\n');
        }
      }
    }
    response.setHeader("Content-Type", "text/plain");
    response.getWriter().write(result.toString());
    // this.logDebug(result.toString());

  }

  /**
   * Logs information. <br/>
   * With this method, you do not need to test is a level is active or not.
   *
   * @param aLevel
   *          a level (Log.XXX)
   * @param arg
   *          what to log, the message with NO '+'
   * @param anExp
   *          an exception if any
   */
  protected final void log(int aLevel, Object... arg) {

    final Log log = LogFactory.getLog(this.getClass());

    if ((aLevel == Priority.DEBUG_INT) && log.isDebugEnabled()) {
      StringBuilder msgBuild = new StringBuilder();
      Throwable error = AbstractObject.aggregateLogMessage(msgBuild, arg);
      String msg = msgBuild.toString();
      log.debug(msg, error);
    } else if ((aLevel == Priority.WARN_INT) && log.isWarnEnabled()) {
      StringBuilder msgBuild = new StringBuilder();
      Throwable error = AbstractObject.aggregateLogMessage(msgBuild, arg);
      String msg = msgBuild.toString();
      log.warn(msg, error);
    } else if ((aLevel == Level.TRACE_INT) && log.isTraceEnabled()) {
      StringBuilder msgBuild = new StringBuilder();
      Throwable error = AbstractObject.aggregateLogMessage(msgBuild, arg);
      String msg = msgBuild.toString();
      log.trace(msg, error);
    } else if ((aLevel == Priority.ERROR_INT) && log.isErrorEnabled()) {
      StringBuilder msgBuild = new StringBuilder();
      Throwable error = AbstractObject.aggregateLogMessage(msgBuild, arg);
      String msg = msgBuild.toString();
      log.error(msg, error);
    } else if ((aLevel == Priority.INFO_INT) && log.isInfoEnabled()) {
      StringBuilder msgBuild = new StringBuilder();
      Throwable error = AbstractObject.aggregateLogMessage(msgBuild, arg);
      String msg = msgBuild.toString();
      log.info(msg, error);
    } else if ((aLevel == Priority.FATAL_INT) && log.isFatalEnabled()) {
      StringBuilder msgBuild = new StringBuilder();
      Throwable error = AbstractObject.aggregateLogMessage(msgBuild, arg);
      String msg = msgBuild.toString();
      log.fatal(msg, error);
    }

  }

  /**
   * Log as fatal.
   *
   * @param arg
   *          what to log, the message with NO '+'
   */
  protected final void logFatal(Object... arg) {
    this.log(Priority.FATAL_INT, arg);
  }

  /**
   * Log as debug.
   *
   * @param arg
   *          what to log, the message with NO '+'
   */
  protected final void logDebug(Object... arg) {
    this.log(Priority.DEBUG_INT, arg);
  }

  /**
   * Log as trace.
   *
   * @param arg
   *          what to log, the message with NO '+'
   */
  protected final void logTrace(Object... arg) {
    this.log(Level.TRACE_INT, arg);
  }

  /**
   * Log as error.
   *
   * @param arg
   *          what to log, the message with NO '+'
   */
  protected final void logError(Object... arg) {
    this.log(Priority.ERROR_INT, arg);
  }

  /**
   * Log as info.
   *
   * @param arg
   *          what to log, the message with NO '+'
   */
  protected final void logInfo(Object... arg) {
    this.log(Priority.INFO_INT, arg);
  }

  /**
   * Log as warn.
   *
   * @param arg
   *          what to log, the message with NO '+'
   */
  protected final void logWarn(Object... arg) {
    this.log(Priority.WARN_INT, arg);
  }

}
