package com.aston.wizards.designpattern;

import java.util.Map;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.ComboDialogField;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;

import com.aston.AstonWizardsPlugin;
import com.aston.LogMessage;
import com.aston.utils.Utils;

/**
 * Wizard page to create a new List class. <p>
 * 
 * A List class is a dynamic array of object, its size can change.
 * 
 * This class is inspired of the org.eclipse.jdt.ui.wizards.NewClassWizardPage.
 * 
 * Part of the <a href="http://renaud91.free.fr/Plugins>Aston Wizard</a><br>
 *
 * @author 
 * <ul>
 * 	<li><a href="mailto:renaud91@free.fr">Ferret Renaud</a></li>
 * </ul>
 * @version 1.0
 * @since Eclipse 2.0
 */
public class NewListCreationWizardPage extends NewArrayCreationWizardPage {
 
	protected final static String SETTINGS_LIST = "list";
	protected ComboDialogField listType;

	/**
	 * Constructor of the object.
	 * 
	 * @param isClass <code>true</code> if a new class is to be created; otherwise
	 * an interface is to be created
	 * @param pageName the wizard page's name
	 * @param templateKey the template key 
	 */
	protected NewListCreationWizardPage(boolean isClass, String pageName, String templateKey) {
		super(isClass, pageName, templateKey);
		LogMessage
				.getInstance()
				.log(
						"-> NewListCreationWizardPage.NewListCreationWizardPage(isClass, pageName, templateKey)");
	}



	/**
	 * Constructor of the object. 
	 */
	public NewListCreationWizardPage() {
		super(true, "NewListCreationWizardPage", "list");
		LogMessage.getInstance().log(
				"-> NewListCreationWizardPage.NewListCreationWizardPage()");
	}

	protected void initThePage() {
		LogMessage.getInstance().log(
				"-> NewListCreationWizardPage.initThePage()");
		super.initThePage();
		setTitle(AstonWizardsPlugin.getResourceString("lg.label.listwizard"));
		setDescription(
			AstonWizardsPlugin.getResourceString("lg.label.listwillgenerate"));
		this.fContainedClassDialogField.setLabelText(
			AstonWizardsPlugin.getResourceString("lg.label.listcontainsonly"));
	}

	/**
	 * Creates the label for the template list.
	 * 
	 * @param composite the parent
	 * @param nColumns the number of columns
	 */
	protected void createControlForTemplate(
		Composite composite,
		int nColumns) {
		LogMessage
				.getInstance()
				.log(
						"-> NewListCreationWizardPage.createControlForTemplate(composite, nColumns)");
		super.createControlForTemplate(composite, nColumns);

		this.listType = new ComboDialogField(SWT.BORDER | SWT.SINGLE);
		this.listType.setLabelText(
			AstonWizardsPlugin.getResourceString("lg.label.listtouse"));
		this.listType.setItems(Utils.asStringArray(Utils.LIST_CLASS));
		this.listType.selectItem(0);
		this.listType.doFillIntoGrid(composite, nColumns);
		this.listType.setDialogFieldListener(this);

		createMethodStubSelectionControls(composite, nColumns);

		setControl(composite);
	}

	/**
	 * Returns type of the list
	 * 
	 * @return the type of the list
	 */
	public String getListClassName() {
		LogMessage.getInstance().log(
				"-> NewListCreationWizardPage.getListClassName()");
		return this.listType.getText();
	}

	/**
	 * Updates status. 
	 */
	protected void doStatusUpdate() {
		LogMessage.getInstance().log(
				"-> NewListCreationWizardPage.doStatusUpdate()");
		super.doStatusUpdate();
		if (this.listType != null) {
			if ("".equals(this.listType.getText().trim())) {
				setErrorMessage(
					AstonWizardsPlugin.getResourceString("lg.warning.listtousembs"));
				setPageComplete(false);
				return;
			}
		}
	}

	/**
	 * Adds specfic values to the Map of keys.
	 * 
	 * @param type type of the object
	 * @param imports the imports of the file
	 * @param monitor the progress bar
	 * @param values the current Map that contains values for the template
	 */
	protected void setSpecificValues(
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor,
		Map values) {
		LogMessage
				.getInstance()
				.log(
						"-> NewListCreationWizardPage.setSpecificValues(type, imports, monitor, values)");
		super.setSpecificValues(type, imports, monitor, values);
		values.put(MARK_LIST_TYPE, getListClassName());
	}
	
	/**
	 * Memorize the selected options. <br>
	 */
	protected void updateSectionValues() {
		LogMessage.getInstance().log(
				"-> NewListCreationWizardPage.updateSectionValues()");
		super.updateSectionValues();
		IDialogSettings section =
			getDialogSettings().getSection(this.getName());
		section.put(SETTINGS_LIST, this.listType.getText());
	}

	/**
	 * Sets the "old" value in the window.
	 */
	protected void loadSelectionValues() {
		LogMessage.getInstance().log(
				"-> NewListCreationWizardPage.loadSelectionValues()");
		super.loadSelectionValues();
		IDialogSettings section =
			getDialogSettings().getSection(this.getName());
		if (section!=null) {			
			String temp = section.get(SETTINGS_LIST);
			
			if (this.listType!=null &&
				temp!=null && 
				!"".equals(temp)) {	
				this.listType.setText(temp);
			}
		}		
	}
	
}