package com.aston;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.internal.core.Openable;
import org.eclipse.jdt.internal.ui.JavaPlugin;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.ComboDialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.DialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.IDialogFieldListener;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.preference.BooleanFieldEditor;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.ContainerSelectionDialog;
import org.eclipse.ui.ide.IDE;

import com.aston.utils.Utils;

/**
 * This is a New Type Wizard.
 * 
 * It simply carries the combo that shows available templates and the text for the file name and path. <br>
 * 
 * This wizard will create a file, but not a class. <br>
 * 
 * Part of the <a href="http://renaud91.free.fr/Plugins>Aston Wizard</a><br>
 *
 * @author 
 * <ul>
 * 	<li><a href="mailto:renaud91@free.fr">Ferret Renaud</a></li>
 * </ul>
 * 
 * @version 1.0
 * @since Eclipse 2.0
 */ 
public abstract class WizardPage
	extends org.eclipse.jface.wizard.WizardPage
	implements IDialogFieldListener, Marks {

	protected final static String SETTINGS_TEMPLATE = "template";
	protected final static String SETTINGS_FILE_PATH = "filePath";

	protected Map templatesMap;
	protected IStructuredSelection selection;

	protected Text fileNameText;
	protected Text filePathText;
	protected ComboDialogField templateList;

	protected BooleanFieldEditor generateBool;

	/**
	 * Constructor for WizardPage.
	 * 
	 * @param pageName the name of the wizard
	 * @param templateKey the template key 
	 */
	public WizardPage(String pageName, String templateKey) {
		super(pageName);
		LogMessage.getInstance().log(
			"-> WizardPage.WizardPage(pageName, templateKey)");
		this.templatesMap =
			AstonWizardsPlugin.getDefault().loadTemplateFor(templateKey);
	}

	/**
	 * Constructor for WizardPage.
	 * 
	 * @param pageName the name of the wizard
	 * @param title the title of the wizard
	 * @param titleImage the image to use for this wizard
	 * @param templateKey the template key 
	 */
	public WizardPage(
		String pageName,
		String title,
		ImageDescriptor titleImage,
		String templateKey) {
		super(pageName, title, titleImage);
		LogMessage.getInstance().log(
			"-> WizardPage.WizardPage(pageName, title, titleImage, templateKey)");
		this.templatesMap =
			AstonWizardsPlugin.getDefault().loadTemplateFor(templateKey);
	}

	/**
	 * The wizard owning this page is responsible for calling this method 
	 * with the current selection. The selection is used to 
	 * initialize the fields of the wizard page.
	 * 
	 * @param selection used to initialize the fields
	 */
	public void init(IStructuredSelection selection) {
		LogMessage.getInstance().log("-> WizardPage.init(selection)");
		this.selection = selection;
	}

	/**
	 * Uses the standard container selection dialog to
	 * choose the new value for the container field.
	 */
	public void handleBrowse() {
		LogMessage.getInstance().log("-> WizardPage.handleBrowse()");
		ContainerSelectionDialog dialog =
			new ContainerSelectionDialog(
				getShell(),
				ResourcesPlugin.getWorkspace().getRoot(),
				false,
				AstonWizardsPlugin.getResourceString("lg.label.selectfilepath"));
		if (dialog.open() == ContainerSelectionDialog.OK) {
			Object[] result = dialog.getResult();
			if (result.length == 1) {
				this.filePathText.setText(String.valueOf(result[0]));
			}
		}
	}

	/**
	 * Returns the file name
	 * 
	 * @return the file name, "" if page is not completly build
	 */
	public String getFileName() {
		LogMessage.getInstance().log("-> WizardPage.getFileName()");
		if (this.fileNameText != null) {
			return this.fileNameText.getText();
		}
		return "";
	}

	/**
	 * Returns the file path used
	 * 
	 * @return the file path used, "" if page is not completly build
	 */
	public String getFilePath() {
		LogMessage.getInstance().log("-> WizardPage.getFilePath()");
		if (this.filePathText != null) {
			return this.filePathText.getText();
		}
		return "";
	}

	/**
	 * Returns the template file name
	 * 
	 * @return the template file name selected
	 */
	public String getTemplateFileName() {
		LogMessage.getInstance().log("-> WizardPage.getTemplateFileName()");
		return (String) this.templatesMap.get(this.templateList.getText());
	}

	/**
	 * The worker method. It will find the container, create the
	 * file if missing or just replace its contents, and open
	 * the editor on the newly created file.
	 * 
	 * @param values a map with all that is needed
	 * @param monitor the monitor
	 * @exception CoreException if problems
	 */
	protected void doFinish(final Map values, IProgressMonitor monitor)
		throws CoreException {
		LogMessage.getInstance().log("-> WizardPage.doFinish(values, monitor)");
		if (this.generateBool == null || this.generateBool.getBooleanValue()) {
			// create a sample file
			monitor.beginTask(
				AstonWizardsPlugin.getResourceString(
					"lg.label.creating",
					(String) values.get(MARK_FILE_NAME)),
				2);
			IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
			IResource resource =
				root.findMember(new Path((String) values.get(MARK_FILE_PATH)));
			if (!resource.exists() || !(resource instanceof IContainer)) {
				Utils.throwCoreException(
					AstonWizardsPlugin.getResourceString(
						"lg.error.pathdoesnotexist",
						(String) values.get(MARK_FILE_PATH)),
					this.getName());
			}
			IContainer container = (IContainer) resource;
			final IFile file =
				container.getFile(
					new Path((String) values.get(MARK_FILE_NAME)));
			InputStream stream = null;					
			try {
				StringBuffer sb =
					AstonWizardsPlugin.getDefault().openContentStream(values);

				stream =
					new ByteArrayInputStream(sb.toString().getBytes());
				if (file.exists()) {
					file.setContents(stream, true, true, monitor);
				} else {
					file.create(stream, true, monitor);
				}
				
			} catch (Exception e) {
				Utils.showError(e, getShell());
				return;
			} finally {
				if (stream!=null) {
					try {
						stream.close();
					} catch (IOException e1) {
					}
				}
			}
			monitor.worked(1);
			monitor.setTaskName(AstonWizardsPlugin.getResourceString("lg.label.open"));
			WizardPage.this.openResource(file);
			updateSectionValues();
			monitor.worked(1);
		}
	}

	/**
	 * Memorize the selected options. <br>
	 */
	protected void updateSectionValues() {
		LogMessage.getInstance().log("-> WizardPage.updateSectionValues()");
		if (getDialogSettings() == null)
			return;
		IDialogSettings section =
			getDialogSettings().getSection(this.getName());
		if (section == null) {
			section = getDialogSettings().addNewSection(this.getName());
		}
		if (this.templateList != null) {
			section.put(SETTINGS_TEMPLATE, this.templateList.getText());
		}
		section.put(SETTINGS_FILE_PATH, this.getFilePath());
	}

	/**
	 * Ensures that all is set correctly
	 */
	protected void dialogChanged() {
		LogMessage.getInstance().log("-> WizardPage.dialogChanged()");
		setPageComplete(true);
		setErrorMessage(null);

		if (this.generateBool == null || this.generateBool.getBooleanValue()) {
			String fileName = getFileName();
			String filePath = getFilePath();

			if (this.fileNameText != null && fileName.trim().length() == 0) {
				setErrorMessage(
					AstonWizardsPlugin.getResourceString(
						"lg.warning.filenamemustbespecified"));
				setPageComplete(false);
				return;
			}

			if (this.filePathText != null && filePath.trim().length() == 0) {
				setErrorMessage(
					AstonWizardsPlugin.getResourceString(
						"lg.warning.filepathmustbespecified"));
				setPageComplete(false);
				return;
			}

			if (this.templateList != null) {
				String template = this.templateList.getText();
				if (template.trim().length() == 0) {
					setErrorMessage(
						AstonWizardsPlugin.getResourceString(
							"lg.warning.templatenamemustbespecified"));
					setPageComplete(false);
					return;
				}
				if (this.templatesMap.get(template.trim()) == null) {
					setErrorMessage(
						AstonWizardsPlugin.getResourceString(
							"lg.warning.templateisnotavailable",
							template));
					setPageComplete(false);
					return;
				}
			}
		}
	}

	/**
	 * Returns the global path.
	 * 
	 * Means, where does the user has decided to make a new file, using 
	 * right click popup menu or toolbar options.
	 * 
	 * @return the path where to create the new file.
	 */
	public String getGlobalPath() {
		LogMessage.getInstance().log("-> WizardPage.getGlobalPath()");

		if (this.selection != null
			&& this.selection.isEmpty() == false
			&& this.selection instanceof IStructuredSelection) {
			IStructuredSelection ssel = (IStructuredSelection) this.selection;
			//			if (ssel.size() > 1) {
			//				return "/";
			//			}
			Object obj = ssel.getFirstElement();
			if (obj instanceof IResource) {
				IContainer container;
				if (obj instanceof IContainer) {
					container = (IContainer) obj;
				} else {
					container = ((IResource) obj).getParent();
				}
				return container.getFullPath().toString();
			} else if (obj instanceof Openable) {
				return ((Openable) obj).getPath().toString();
				//			} else  {
				//				System.out.println(obj.getClass());
			}
		}
		//System.out.println("WizardPage.getGlobalPath returns /");
		return "/";
	}

	/**
	 * Sets the "old" value in the window.
	 */
	protected void loadSectionValues() {
		LogMessage.getInstance().log("-> WizardPage.loadSectionValues()");
		if (getDialogSettings() == null)
			return;
		IDialogSettings section =
			getDialogSettings().getSection(this.getName());

		if (section != null) {
			String temp = section.get(SETTINGS_TEMPLATE);
			// Path is not keep because if we are in an other project 
			// it can bring some unwanted behaviour
			// String path = section.get(SETTINGS_FILE_PATH);

			String path = getGlobalPath();

			if (this.templatesMap != null
				&& this.templateList != null
				&& temp != null
				&& !"".equals(temp)) {
				this.templateList.setText(temp);
			}
			if (this.filePathText != null
				&& path != null
				&& !"".equals(path)) {
				this.filePathText.setText(path);
			}
		}
	}

	/**
	 * Makes the page visible. 
	 * 
	 * @param visible true will show the page, false will hide it
	 */
	public void setVisible(boolean isVisible) {
		LogMessage.getInstance().log("-> WizardPage.setVisible(isVisible)");
		if (isVisible) {
			this.loadSectionValues();
		}

		super.setVisible(isVisible);
	}

	/**
	 * Builds the window.
	 * 
	 * @param parent the parent of this component
	 */
	public abstract void createControl(Composite parent);

	/**
	 * Gets a runnable object that will be in charge of creating the file
	 * 
	 * @return a runnable object that will be in charge of creating the file
	 */
	public abstract IRunnableWithProgress getRunnable();

	/**
	 * Creates the label for the template list.
	 * 
	 * @param composite the parent
	 * @param nColumns the number of columns
	 */
	protected void createControlForTemplate(
		Composite composite,
		int nColumns) {
		LogMessage.getInstance().log(
			"-> WizardPage.createControlForTemplate(composite, nColumns)");
		this.templateList =
			new ComboDialogField(SWT.READ_ONLY | SWT.BORDER | SWT.SINGLE);
		this.templateList.setItems(
			Utils.asStringArray(this.templatesMap, true));
		this.templateList.selectItem(0);
		this.templateList.setLabelText(
			AstonWizardsPlugin.getResourceString("lg.label.temptouse"));
		this.templateList.doFillIntoGrid(composite, nColumns);
		this.templateList.setDialogFieldListener(this);
	}

	/**
	 * What to do if field change.
	 * 
	 * @param field who has changed
	 */
	public void dialogFieldChanged(DialogField field) {
		LogMessage.getInstance().log("-> WizardPage.dialogFieldChanged(field)");
		dialogChanged();
	}

	/**
	 * This object will be in charge of creating the new file.
	 */
	protected final static class AstonRunnableWithProgress
		implements IRunnableWithProgress {

		protected final Map values;
		protected final com.aston.WizardPage page;

		public AstonRunnableWithProgress(
			final Map someValues,
			final com.aston.WizardPage aPage) {
			super();
			LogMessage.getInstance().log(
				"-> AstonRunnableWithProgress.AstonRunnableWithProgress(someValues, aPage)");
			this.values = someValues;
			this.page = aPage;
		}

		public void run(IProgressMonitor monitor)
			throws InvocationTargetException, InterruptedException {
			LogMessage.getInstance().log(
				"-> AstonRunnableWithProgress.run(monitor)");
			if (monitor == null) {
				monitor = new NullProgressMonitor();
			}
			try {
				this.page.doFinish(this.values, monitor);
			} catch (CoreException e) {
				Utils.showError(e, this.page.getShell());
				return;
			}
			this.delete();
		}

		protected void delete() {
			LogMessage.getInstance().log(
				"-> AstonRunnableWithProgress.delete()");
			if (this.values != null) {
				this.values.clear();
			}
		}
	}

	protected final void openResource(final IResource file) {
		LogMessage.getInstance().log("-> NewCreationWizard.openResource(file)");
		if (getShell() != null && file.getType() == IResource.FILE) {
			getShell().getDisplay().asyncExec(new Runnable() {
				public void run() {
					IWorkbenchPage page =
						PlatformUI
							.getWorkbench()
							.getActiveWorkbenchWindow()
							.getActivePage();
					try {
						IDE.openEditor(page, (IFile)file, true);						
						// For Eclipse 2.x : page.openEditor((IFile) file);
					} catch (PartInitException e) {
						JavaPlugin.log(e);
					}
				}
			});
		}
	}

}
