package com.aston;

import java.awt.Point;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.search.IJavaSearchConstants;
import org.eclipse.jdt.core.search.IJavaSearchScope;
import org.eclipse.jdt.core.search.SearchEngine;
import org.eclipse.jdt.internal.ui.dialogs.TypeSelectionDialog;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.DialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.IDialogFieldListener;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.IListAdapter;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.IStringButtonAdapter;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.LayoutUtil;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.ListDialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.StringButtonDialogField;
import org.eclipse.jdt.internal.ui.wizards.dialogfields.StringDialogField;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;

import com.aston.utils.Utils;

/**
 * This is a New NewTypeContainer Wizard.
 * 
 * This wizard will carry an object inside, like an array or a List of object. <br>
 * 
 * This class also contains a lot of methods that make replacement in templates file. <br>
 * 
 * This wizard will create a Java Class. <br>
 * 
 * Part of the <a href="http://renaud91.free.fr/Plugins>Aston Wizard</a><br>
 *
 * @author 
 * <ul>
 * 	<li><a href="mailto:renaud91@free.fr">Ferret Renaud</a></li>
 * </ul>
 * 
 * @version 1.0
 * @since Eclipse 2.0 
 */
public abstract class NewTypeContainerWizardPage
	extends com.aston.NewTypeWizardPage {

	protected StringButtonDialogField fContainedClassDialogField;
	protected IType containedType;

	/**
	 * Constructor of the object.
	 * 
	 * @param isClass <code>true</code> if a new class is to be created; otherwise
	 * an interface is to be created
	 * @param pageName the wizard page's name
	 * @param templateKey the template key
	 */
	public NewTypeContainerWizardPage(boolean isClass, String pageName, String templateKey) {
		super(isClass, pageName, templateKey);
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeContainerWizardPage.NewTypeContainerWizardPage(isClass, pageName, templateKey)");
	}

	/**
	 * Updates status. 
	 * 
	 * This method is called each time a change in made in the fields of the wizard panel
	 */
	protected void doStatusUpdate() {
		LogMessage.getInstance().log(
				"-> NewTypeContainerWizardPage.doStatusUpdate()");
		super.doStatusUpdate();
		if (isPageComplete()) {
			if ("".equals(this.fContainedClassDialogField.getText())) {
				setErrorMessage(
					AstonWizardsPlugin.getResourceString(
						"lg.warning.objmustbespecified"));
				setPageComplete(false);
				return;
			}
		}
	}

	/**
	 * Adds specfic values to the Map of keys.
	 * 
	 * @param type type of the object
	 * @param imports the imports of the file
	 * @param monitor the progress bar
	 * @param values the current Map that contains values for the template
	 */
	protected void setSpecificValues(
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor,
		Map values) {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeContainerWizardPage.setSpecificValues(type, imports, monitor, values)");
		String inType = Utils.getClassName(getContainedClassName());

		values.put(MARK_CONTAINED_TYPE, inType);
		values.put(MARK_CONTAINED_TYPE_AS_NULL, Utils.getNull(inType));
		values.put(MARK_CONTAINED_TYPE_CLASS_NAME, Utils.getObject(inType));
	}

	/**
	 * Creates the controls for the superclass name field. Expects a <code>GridLayout</code> 
	 * with at least 3 columns. 
	 * 
	 * @param composite the parent composite
	 * @param nColumns number of columns to span
	 */
	protected void createContainedClassControls(
		Composite composite,
		int nColumns) {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeContainerWizardPage.createContainedClassControls(composite, nColumns)");
		if (this.fContainedClassDialogField != null) {
			this.fContainedClassDialogField.doFillIntoGrid(composite, nColumns);
			LayoutUtil.setWidthHint(
					this.fContainedClassDialogField.getTextControl(null),
				getMaxFieldWidth());
		}
	}

	/**
	 * Creates the window that will represent the wizard page. 
	 * 
	 * @param parent the parent of this page
	 */
	public void createControl(Composite parent) {
		LogMessage.getInstance().log(
				"-> NewTypeContainerWizardPage.createControl(parent)");
		initializeDialogUnits(parent);

		Composite composite = new Composite(parent, SWT.NONE);

		int nColumns = 4;

		GridLayout layout = new GridLayout();
		layout.numColumns = nColumns;
		composite.setLayout(layout);

		// pick & choose the wanted UI components

		createContainerControls(composite, nColumns);
		createPackageControls(composite, nColumns);
		createEnclosingTypeControls(composite, nColumns);

		createSeparator(composite, nColumns);

		createTypeNameControls(composite, nColumns);
		createModifierControls(composite, nColumns);

		createSuperClassControls(composite, nColumns);
		createContainedClassControls(composite, nColumns);
		createSuperInterfacesControls(composite, nColumns);

		createControlForTemplate(composite, nColumns);

		createMethodStubSelectionControls(composite, nColumns);

		setControl(composite);
	}

	/**
	 * Generates the java file. 
	 * 
	 * @param type type of the object
	 * @param imports the imports of the file
	 * @param monitor the progress bar
	 * @param values a Map that contains every information needed for the creation of the class
	 * 
	 * @exception CoreException if an error occurred
	 */
	protected void createInternal(
		IType type,
		ImportsManager imports,
		IProgressMonitor monitor,
		Map values)
		throws CoreException {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeContainerWizardPage.createInternal(type, imports, monitor, values)");
		StringBuffer fileSb = null;

		try {
			fileSb = AstonWizardsPlugin.getDefault().openContentStream(values);
		} catch (IOException e) {
			Utils.showError(e, getShell());
			return;
		}

		createInheritedMethods(
			type,
			false,
			isCreateInherited(),
			imports,
			monitor);

		if (this.getContainedClassName().indexOf(".") != -1) {
			imports.addImport(this.getContainedClassName());
		}

		transformBasicToObject(
			fileSb,
			(String) values.get(MARK_CONTAINED_TYPE),
			MARK_CONTAINED_TYPE_AS_OBJECT);
		transformObjectToBasic(
			fileSb,
			(String) values.get(MARK_CONTAINED_TYPE),
			MARK_CONTAINED_TYPE_AS_VALUE);

		String file = fileSb.toString();

		createImports(file, type, imports, monitor);
		createFields(file, type, imports, monitor);
		if (isCreateConstructors()) {
			createConstructor(file, type, imports, monitor);
		}
		createMethods(file, type, imports, monitor);
		createInerType(file, type, imports, monitor);

		updateSectionValues();

		if (monitor != null) {
			monitor.done();
		}
	}

	/**
	 * Returns type that will be contained in the class.
	 * 
	 * @return the element name, or null if page is not yet created
	 */
	public String getContainedClassName() {
		LogMessage.getInstance().log(
				"-> NewTypeContainerWizardPage.getContainedClassName()");
		if (this.fContainedClassDialogField != null) {
			return this.fContainedClassDialogField.getText();
		}
		return null;
	}

	/**
	 * Transform basic object into their wrapped form.
	 * 
	 * @param file the StringBuffer loaded from the template file
	 * @param basic the basic type contained in the class, should be "int", "float" ...
	 * @param aAsObjectMark the mark to intercept (like MARK_CONTAINED_KEY_AS_OBJECT_)
	 */
	protected void transformBasicToObject(
		StringBuffer file,
		String basic,
		String aAsObjectMark) {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeContainerWizardPage.transformBasicToObject(file, basic, aAsObjectMark)");
		if (basic == null)
			return;
		// boolean isBasic = Utils.isBasicType(basic);
		String typeStart = "<".concat(aAsObjectMark);
		String typeEnd = "</".concat(aAsObjectMark);
		// Search of where to make replacement
		String sFile = file.toString();
		List where = new ArrayList();
		int start = sFile.indexOf(typeStart, 0);
		int end = -1;
		while (start != -1) {
			end = sFile.indexOf(typeEnd, start + typeStart.length());
			if (end == -1) {
				Utils.showError(
					AstonWizardsPlugin.getResourceString(
						"lg.error.missingendtag",
						aAsObjectMark),
					getShell());
				return;
			}
			Point last = new Point(start, end);
			where.add(last);
			start = sFile.indexOf(typeStart, end + typeEnd.length());
		}
		// Start to replace, begins by the end of the string
		for (int i = where.size() - 1; i >= 0; i--) {
			Point p = (Point) where.get(i);
			int startR = sFile.indexOf(">", p.x + typeStart.length()) + 1;
			int endR = sFile.indexOf(">", p.y + typeEnd.length()) + 1;
			String paramName = sFile.substring(startR, p.y);
			Utils.asNewObject(basic, paramName, file, p.x, endR);
			p = null;
		}
		where.clear();
		where = null;
	}

	/**
	 * Transform wrapped object into their non form.
	 * 
	 * @param file the StringBuffer loaded from the template file
	 * @param basic the basic type contained in the class, should be "int", "float" ...
	 * @param aAsValueMark the mark to intercept (like MARK_CONTAINED_KEY_AS_VALUE_)
	 */
	protected void transformObjectToBasic(
		StringBuffer file,
		String basic,
		String aAsValueMark) {
		LogMessage
				.getInstance()
				.log(
						"-> NewTypeContainerWizardPage.transformObjectToBasic(file, basic, aAsValueMark)");
		if (basic == null)
			return;
		boolean isBasic = Utils.isBasicType(basic);
		String typeStart = "<".concat(aAsValueMark);
		String typeEnd = "</".concat(aAsValueMark);
		// Search of where to make replacement
		String sFile = file.toString();
		List where = new ArrayList();
		int start = sFile.indexOf(typeStart, 0);
		int end = -1;
		while (start != -1) {

			end = sFile.indexOf(typeEnd, start);
			if (end == -1) {
				Utils.showError(
					AstonWizardsPlugin.getResourceString(
						"lg.error.missingendtag",
						aAsValueMark),
					getShell());
				return;
			}
			Point last = new Point(start, end);
			where.add(last);
			start = sFile.indexOf(typeStart, end);
		}
		// Start to replace, begins by the end of the string
		for (int i = where.size() - 1; i >= 0; i--) {
			Point p = (Point) where.get(i);
			int startR = sFile.indexOf(">", p.x + typeStart.length()) + 1;
			int endR = sFile.indexOf(">", p.y + typeEnd.length()) + 1;
			String paramName = sFile.substring(startR, p.y);
			if (isBasic) {
				file.replace(p.x, endR, paramName + "." + basic + "Value()");
			} else {
				file.replace(p.x, endR, paramName);
			}
			p = null;
		}
		where.clear();
		where = null;
	}

	/**
	 * Inner class used when the user click on the button that will show him all available objects.
	 */
	protected class TypeFieldsAdapter
		implements IStringButtonAdapter, IDialogFieldListener, IListAdapter {
		protected String dialogTitle;
		protected String dialogMessage;
		protected int javaSearchConstants;

		public TypeFieldsAdapter() {
			super();
			LogMessage.getInstance().log(
					"-> TypeFieldsAdapter.TypeFieldsAdapter()");
			this.dialogTitle =
				AstonWizardsPlugin.getResourceString("lg.label.selectobject");
			this.dialogMessage =
				AstonWizardsPlugin.getResourceString("lg.label.selectobject");
			this.javaSearchConstants = IJavaSearchConstants.TYPE;
		}

		public void changeControlPressed(DialogField field) {
			LogMessage.getInstance().log(
					"-> TypeFieldsAdapter.changeControlPressed(field)");
			IPackageFragmentRoot root = getPackageFragmentRoot();
			if (root == null) {
				return;
			}

			IJavaElement[] elements =
				new IJavaElement[] { root.getJavaProject()};
			IJavaSearchScope scope =
				SearchEngine.createJavaSearchScope(elements);

			TypeSelectionDialog dialog =
				new TypeSelectionDialog(
					getShell(),
					getWizard().getContainer(),
					this.javaSearchConstants,
					scope);
			dialog.setTitle(this.dialogTitle); // "Choose contained object type"
			dialog.setMessage(this.dialogMessage);
			// "Select the type of object contained in this object"

			if (dialog.open() == TypeSelectionDialog.OK) {
				NewTypeContainerWizardPage.this.containedType = (IType) dialog.getFirstResult();
				if (field != null) {
					((StringDialogField) field).setText(	
						NewTypeContainerWizardPage.this.containedType.getFullyQualifiedName());
				}
			}
		}

		/**
		 * Does nothing, but used for Eclipse 2.1 compatibility
		 * 
		 * @param field a field
		 * @param index an index
		 */
		public void customButtonPressed(DialogField field, int index) {
			LogMessage.getInstance().log(
					"-> TypeFieldsAdapter.customButtonPressed(field, index)");
			if (field != null && index != 0);
		}

		/**
		 * Does nothing, but used for Eclipse 2.1 compatibility
		 * 
		 * @param field a field
		 */
		public void selectionChanged(DialogField field) {
			LogMessage.getInstance().log(
					"-> TypeFieldsAdapter.selectionChanged(field)");
			if (field != null);
		}

		/**
		 * Does nothing, but used for Eclipse 2.1 compatibility
		 * 
		 * @param field a field
		 */
		public void dialogFieldChanged(DialogField field) {
			LogMessage.getInstance().log(
					"-> TypeFieldsAdapter.dialogFieldChanged(field)");
			doStatusUpdate();
		}

		/**
		 * Sets the dialogMessage.
		 * 
		 * @param dialogMessage The dialogMessage to set
		 */
		public void setDialogMessage(String dialogMessage) {
			LogMessage.getInstance().log(
					"-> TypeFieldsAdapter.setDialogMessage(dialogMessage)");
			this.dialogMessage = dialogMessage;
		}

		/**
		 * Sets the dialogTitle.
		 * 
		 * @param dialogTitle The dialogTitle to set
		 */
		public void setDialogTitle(String dialogTitle) {
			LogMessage.getInstance().log(
					"-> TypeFieldsAdapter.setDialogTitle(dialogTitle)");
			this.dialogTitle = dialogTitle;
		}

		/**
		 * Sets the javaSearchConstants.
		 * @param javaSearchConstants The javaSearchConstants to set
		 */
		public void setJavaSearchConstants(int javaSearchConstants) {
			LogMessage
					.getInstance()
					.log(
							"-> TypeFieldsAdapter.setJavaSearchConstants(javaSearchConstants)");
			this.javaSearchConstants = javaSearchConstants;
		}

		/**
		 * Does nothing, but used for Eclipse 2.1 compatibility
		 * 
		 * @param field a field
		 * @param index an index
		 */
		public void customButtonPressed(ListDialogField field, int index) {
			LogMessage.getInstance().log(
					"-> TypeFieldsAdapter.customButtonPressed(field, index)");
		}

		/**
		 * Does nothing, but used for Eclipse 2.1 compatibility
		 * 
		 * @param field a field  
		 */
		public void doubleClicked(ListDialogField field) {
			LogMessage.getInstance().log(
					"-> TypeFieldsAdapter.doubleClicked(field)");
		}

		/**
		 * Does nothing, but used for Eclipse 2.1 compatibility
		 * 
		 * @param field a field  
		 */
		public void selectionChanged(ListDialogField field) {
			LogMessage.getInstance().log(
					"-> TypeFieldsAdapter.selectionChanged(field)");
		}

	}
}