package com.aston;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.TreeMap;

import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.jdt.internal.ui.JavaPlugin;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

import com.aston.utils.MessageFormat;
import com.aston.utils.Utils;

/**
 * The main plugin class to be used in the desktop.
 * 
 * This plugin handles all aston wizard :
 * 
 * Part of the <a href="http://renaud91.free.fr/Plugins>Aston Wizard</a><br>
 *
 * @author 
 * <ul>
 * 	<li><a href="mailto:renaud91@free.fr">Ferret Renaud</a></li>
 * </ul>
 * @version 3.0
 * @since Eclipse 3.0 
 */
public final class AstonWizardsPlugin extends AbstractUIPlugin {
	/** The shared instance. */
	private static AstonWizardsPlugin plugin;
	/** Name of the preference file for template. */
	private final static String TEMPLATE_PROPERTIES = "templates";
	/** Name of the property file for language. */
	private final static String LANGUAGE_PROPERTIES = "language";
	/** Name of the property file for language. */
	private static ResourceBundle LANGUAGE_BUNDLE;
	
	/** Where are the templates. */
	private static Preferences TEMPLATE_BUNDLE;

	/**
	 * The constructor.
	 */
	public AstonWizardsPlugin() {
		super();	
		this.doInit();
	}

	private final void doInit() {
		LogMessage.getInstance().log("-> AstonWizardsPlugin.doInit()");
		plugin = this;
		try {
			// Language will be a Bundle
			AstonWizardsPlugin.LANGUAGE_BUNDLE =
				ResourceBundle.getBundle(LANGUAGE_PROPERTIES);
		} catch (MissingResourceException e) {
			Utils.showError(
				"File that contains language (language properties) was not found !",
				JavaPlugin.getActiveWorkbenchShell());
		}
	}
	
	/**
	 * This method is called upon plug-in activation
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
		try {
			TEMPLATE_BUNDLE = new Preferences();
			InputStream is = super.openStream(new Path(getLocaleLanguage(TEMPLATE_PROPERTIES)));
			TEMPLATE_BUNDLE.load(is);
		} catch (Exception e) {
			Utils.showError(
				"File that contains language (language properties) was not found !",
				JavaPlugin.getActiveWorkbenchShell());
		}
	}

	/**
	 * This method is called when the plug-in is stopped
	 */
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
	}

	/**
	 * Returns the shared instance.
	 * 
	 * @return the shared instance.
	 */
	public static AstonWizardsPlugin getDefault() {
		LogMessage.getInstance().log("-> AstonWizardsPlugin.getDefault()");
		return plugin;
	}

	/**
	 * Returns the workspace instance.
	 * 
	 * @return the workspace instance.
	 */
	public static IWorkspace getWorkspace() {
		LogMessage.getInstance().log("-> AstonWizardsPlugin.getWorkspace()");
		return ResourcesPlugin.getWorkspace();
	}

	/**
	 * Loads a template file and build a string from it. <br>
	 * 
	 * @param aTemplateName the file name of the template
	 * 
	 * @return a String that represent the template file
	 * @exception IOException if the file was not found or if a problem occured while reading the template file
	 */
	public final StringBuffer buildFromTemplate(String aTemplateName)
		throws IOException {
		LogMessage.getInstance().log(
				"-> AstonWizardsPlugin.buildFromTemplate(aTemplateName)");
		StringBuffer sb = new StringBuffer(128);
		final String newLine = System.getProperty("line.separator");
		BufferedReader br = null;
		InputStreamReader isr = null;
		InputStream in = null;
		try {
			in =
				AstonWizardsPlugin.getDefault().openStream(new Path(aTemplateName));
			isr = new InputStreamReader(in);
			br = new BufferedReader(isr);
			String aLine = br.readLine();
			while (aLine != null) {
				if (aLine.length() > 0 && aLine.charAt(0) == '#') {
					// Does nothing
				} else {
					sb.append(aLine);
					sb.append(newLine);
				}
				aLine = br.readLine();
			}
		} catch (IOException e) {
			throw e;
		} finally {
			try {
				if (br!=null) br.close();
			} catch (IOException e1) {
			}
			try {
				if (isr!=null) isr.close();
			} catch (IOException e2) {
			}
			try {
				if (in!=null) in.close();
			} catch (IOException e3) {
			}
		}
		return sb;
	}

	/**
	 * Loads all templates that starts with the given string.
	 * 
	 * Description will be used as key, and file name will be the value.
	 * 
	 * @param nameBeginsWith a starting entry in the properties file. If null
	 * returns an empty map
	 * 
	 * @return a map of all templates
	 */
	public final Map loadTemplateFor(String nameBeginsWith) {
		LogMessage.getInstance().log(
				"-> AstonWizardsPlugin.loadTemplateFor(nameBeginsWith)");
		int i = 1;
		Map m = new TreeMap(); // TreeMap is slow but keeps order
		if (nameBeginsWith==null) {
			return m;
		}

		while (true) {
			String key = nameBeginsWith + ".template." + i;
			String fileName = TEMPLATE_BUNDLE.getString(key);
			if ("".equals(fileName)) {
				break;
			}
			String description = TEMPLATE_BUNDLE.getString(key + ".label");
			if ("".equals(description)) {
				description = fileName;
			}
			//TODO I presume here that there will not be more than 99 templates for one panel
			StringBuffer mapKey = new StringBuffer(128); 
			if (i<10) {
				mapKey.append('0').append(i);
			} else {
				mapKey.append(i);
			}
			mapKey.append("] ").append(description);
			// Adding the number i]  will keep order of template declaration
			m.put(mapKey.toString(), fileName);
			mapKey = null;
			key = null;
			fileName = null;
			description = null;
			i++;
		}
		return m;
	}

	/**
	 * This method will fill the gap in the template files.
	 * 
	 * One of the key in the map must be the name of the template file (Marks.MARK_TEMPLATE_NAME)
	 * 
	 * @param values a map of (key,value), where key is a MARK_..., and value a String
	 * 
	 * @return the input stream that will be used for the creation of the file
	 * @exception IOException if the template file was not found or if a problem occured while reading the template file
	 */
	public final StringBuffer openContentStream(final Map values)
		throws IOException, CoreException {
		LogMessage.getInstance().log(
				"-> AstonWizardsPlugin.openContentStream(values)");
		String tName = (String) values.get(Marks.MARK_TEMPLATE_NAME);

		StringBuffer res = res = this.buildFromTemplate(tName);

		Iterator it = values.keySet().iterator();
		while (it.hasNext()) {
			String key = (String) it.next();
			if (key != Marks.MARK_TEMPLATE_NAME
				&& key != Marks.MARK_JSP_DESCRIPTION
				&& key != Marks.MARK_JSP_DISPLAY_NAME
				&& key != Marks.MARK_JSP_KEY_NAME
				&& key != Marks.MARK_JSP_MAPPING_URL
				&& key != Marks.MARK_FILE_NAME
				&& key != Marks.MARK_FILE_PATH) {
				String value = (String) values.get(key);
				Utils.replaceWithForEmptyXmlTokenType(res, key, value);
				value = null;
			}
			key = null;
		}
		it = null;
		return res;
	}

	/**
	 * Returns a key from the properties file.
	 * 
	 * @param aKey the key to find
	 * @return the string attached to the key
	 */
	public final static String getResourceString(String aKey) {
		LogMessage.getInstance().log("-> AstonWizardsPlugin.getString(aKey)");
		return AstonWizardsPlugin.LANGUAGE_BUNDLE.getString(aKey);
	}

	/**
	 * Returns a key from the properties file.
	 * 
	 * @param aKey the key to find
	 * @param arg0 a hole in the string
	 * 
	 * @return the string attached to the key
	 */
	public final static String getResourceString(String aKey, String arg0) {
		LogMessage.getInstance().log(
				"-> AstonWizardsPlugin.getString(aKey, arg0)");
		return MessageFormat.format(
			AstonWizardsPlugin.LANGUAGE_BUNDLE.getString(aKey),
			new String[] { arg0 });
	}

	/**
	 * Returns a key from the properties file.
	 * 
	 * @param aKey the key to find
	 * @param arg0 a hole in the string
	 * @param arg1 second hole in the string
	 * 
	 * @return the string attached to the key
	 */
	public final static String getResourceString(
		String aKey,
		String arg0,
		String arg1) {
		LogMessage.getInstance().log(
				"-> AstonWizardsPlugin.getString(aKey, arg0, arg1)");
		return MessageFormat.format(
			AstonWizardsPlugin.LANGUAGE_BUNDLE.getString(aKey),
			new String[] { arg0, arg1 });
	}

	/**
	 * Gets the language according to the current locale.
	 * 
	 * @param defaultName the default name of propertie file
	 * 
	 * @return the name of the propertie file with specific locale
	 */
	private String getLocaleLanguage(String defaultName) {
		LogMessage.getInstance().log(
				"-> AstonWizardsPlugin.getLocaleLanguage(defaultName)");
		//	String country = Locale.getDefault().getCountry();
		String language = Locale.getDefault().getLanguage();

		StringBuffer sb = new StringBuffer(defaultName);
		sb.append("_").append(language);
		sb.append(".properties");
		// Search for file if present
		try {			
			new InputStreamReader(this.openStream(new Path(sb.toString())));
		} catch (IOException e) {
			// Not present
			return defaultName.concat(".properties");
		} catch(NullPointerException npe) {
			// Should happen when template files are not in visible path
			Utils.showError(
					AstonWizardsPlugin.LANGUAGE_BUNDLE.getString(
						"lg.error.tpfileNotFound"),
					JavaPlugin.getActiveWorkbenchShell());
		}
		return sb.toString();
	}
	
	/**
	 * Returns the current shell
	 * 
	 * @return the current shell
	 */
	public Shell getShell() {
		LogMessage.getInstance().log("-> AstonWizardsPlugin.getShell()");
		return super.getWorkbench().getWorkbenchWindows()[0].getShell();
	}
	
	/**
	 * Returns the plugin's resource bundle,
	 */
	public ResourceBundle getLanguageBundle() {
		return LANGUAGE_BUNDLE;
	}
	
	/**
	 * Returns the plugin's template preferences
	 */
	public static Preferences getTemplatePreferences() {
		return TEMPLATE_BUNDLE;
	}

}
