package fr.lifl.Aquarium.Simulation;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.awt.Point;
import java.awt.Dimension;
import java.awt.Color;
import java.io.Serializable;

/**
  * <i>Classe</i> permettant de definir l'objet <tt>Ver</tt>. <P>
  *
  * Cette classe contient les methodes permettant au ver de se deplacer. <P>
  *
  * Derniere modification : 27 Fevrier 2000
  *
  * @author
  * <ul>
  *   <li>Hiblot Sebastien (V1.00)
  *   <li><a href="mailto:muriel_vidonne@yahoo.fr">Vidonne Muriel</a> (V1.10) (V2.2)
  *   <li><a href="mailto:renaud91@hotmail.com">Ferret Renaud</a> (V2.++)
  * </ul>
  *
  * @version 2.4
  *
  * @see fr.lifl.Aquarium.Simulation.Genotype
  * @see fr.lifl.Aquarium.Simulation.ListeVer
  */
public class Ver implements Serializable,Cloneable
{
  
  /**
    * Methode qui se charge de deplacer le <tt>Ver</tt>. <p>
    *
    * Le deplacement d'un <tt>Ver</tt> est fonction de ce 
    * qu'il voit et de son <tt>Genotype</tt>. <p>
    *
    * @param tn la <tt>TableNourriture</tt>
    * @param piste la <tt>PisteGraphique</tt>
    * @param coutUnMouvement le cout du deplacement
    * @param coeffAjustement le coefficient a appliquer lors de l'action <i>voir</i>
    */
  public void deplacer(TableNourriture tn, PisteGraphique piste, float coutUnMouvement, float coeffAjustement)
  {
    int i;
    String vision = null;
    
    // Ce que voit le ver
    vision = vision(DIAGONALE, tn, piste.getPiste(), coeffAjustement);
    
    // Quel mouvement lui est associe
    String deplacement = genotype.getAdn().getValeurGene(vision);
    
    // System.out.println(vision+","+deplacement+","+direction);
    
    // System.out.println("Je vois=" + vision + ", je fais=" + deplacement + ", ma direction est =" + direction);
    if(deplacement == null)
    {
      System.err.println(Erreur.getString("erreur_ver_deplacer"));
      return ;
    }
    Point tete = getTete();
    for(i = 0;i < deplacement.length();i++)
    {
      char sens = deplacement.charAt(i);
      switch(sens)
      {
        
        // Je vais en FACE par rapport a ma direction
        case FACE:
          switch(direction)
          {
            case FACE:
              if(bouger(tete.x, tete.y - 1, tn, coutUnMouvement, piste))
              {
                direction = FACE;
                return ;
              }
              else
                break;
            
            case BAS:
              if(bouger(tete.x, tete.y + 1, tn, coutUnMouvement, piste))
              {
                direction = BAS;
                return ;
              }
              else
                break;
            
            case DROITE:
              if(bouger(tete.x + 1, tete.y, tn, coutUnMouvement, piste))
              {
                direction = DROITE;
                return ;
              }
              else
                break;
            
            case GAUCHE:
              if(bouger(tete.x - 1, tete.y, tn, coutUnMouvement, piste))
              {
                direction = GAUCHE;
                return ;
              }
              else
                break;
          }
          break;
        
        case DROITE:
          switch(direction)
          {
            case FACE:
              if(bouger(tete.x + 1, tete.y, tn, coutUnMouvement, piste))
              {
                direction = DROITE;
                return ;
              }
              else
                break;
            
            case BAS:
              if(bouger(tete.x - 1, tete.y, tn, coutUnMouvement, piste))
              {
                direction = GAUCHE;
                return ;
              }
              else
                break;
            
            case DROITE:
              if(bouger(tete.x, tete.y + 1, tn, coutUnMouvement, piste))
              {
                direction = BAS;
                return ;
              }
              else
                break;
            
            case GAUCHE:
              if(bouger(tete.x, tete.y - 1, tn, coutUnMouvement, piste))
              {
                direction = FACE;
                return ;
              }
              else
                break;
          }
          break;
        
        case GAUCHE:
          switch(direction)
          {
            case FACE:
              if(bouger(tete.x - 1, tete.y, tn, coutUnMouvement, piste))
              {
                direction = GAUCHE;
                return ;
              }
              else
                break;
            
            case BAS:
              if(bouger(tete.x + 1, tete.y, tn, coutUnMouvement, piste))
              {
                direction = DROITE;
                return ;
              }
              else
                break;
            
            case DROITE:
              if(bouger(tete.x, tete.y - 1, tn, coutUnMouvement, piste))
              {
                direction = FACE;
                return ;
              }
              else
                break;
            
            case GAUCHE:
              if(bouger(tete.x, tete.y + 1, tn, coutUnMouvement, piste))
              {
                direction = BAS;
                return ;
              }
              else
                break;
          }
      }
    } // fin du for
  
  // Si j'arrive ici, c'est que je n'ai pas put me deplacer
  }
  
  /**
    * Methode qui tue le <tt>Ver</tt> proprement. <p>
    *
    * @param piste la piste  
    */
  public void tuerVer(PisteGraphique piste)
  {
    int i;
    Point p = null;
    for(i = 0;i < position.size();i++)
    {
      p = (Point)position.elementAt(i);
      piste.setCaseVide(p);
    }
    position.removeAllElements();
  }
  
  /**
    * Transforme un objet <tt>Ver</tt> en une <tt>String</tt>. <p>
    *
    * @return une <tt>String</tt> representant un <tt>Ver</tt>
    */
  public String toString()
  {
    StringBuffer resu = new StringBuffer();
    resu.append("\n[ ");
    resu.append(Langage.getString("vie"));
    resu.append("=");
    resu.append(vie);
    resu.append(", ");
    resu.append(Langage.getString("generation"));
    resu.append("=");
    resu.append(nbGeneration);
    resu.append(",\n");
    if(getTete() != null)
    {
      resu.append(Langage.getString("tete"));
      resu.append("=(");
      resu.append(getTete().x);
      resu.append(", ");
      resu.append(getTete().y);
      resu.append("),\n");
    }
    resu.append(Langage.getString("genotype"));
    resu.append("=");
    resu.append(genotype);
    resu.append("\n]");
    return resu.toString();
  }
  
  /**
    * Constructeur d'un objet <tt>Ver</tt>. <p>
    *
    * <b>ATTENTION</b> : le <tt>Ver</tt> est directement ajoute a la
    * piste. <p>
    *
    * @param piste la <tt>PisteGraphique</tt> auquel il appartient
    * @param cas le cas etudie
    * @param taille la taille du <tt>Ver</tt>
    *
    * @see fr.lifl.Aquarium.Simulation.Genotype
    * @see fr.lifl.Aquarium.Simulation.Adn
    */
  public Ver(PisteGraphique piste, String cas, int taille) 
  {
    this(piste, null, cas, taille, false, null);
  }
  
  /**
    * Constructeur d'un objet <tt>Ver</tt>. <p>
    *
    * <b>ATTENTION</b> : le <tt>Ver</tt> est directement ajoute a la
    * piste. <p>
    *
    * @param piste la <tt>PisteGraphique</tt> auquel il appartient  
    * @param tete la coordonnee de sa tete
    * @param cas le cas etudie
    * @param taille la taille du <tt>Ver</tt>
    * @param parfait
    * <ul>
    *   <li>true : l'<tt>Adn</tt>, contenu dans le <tt>Genotype</tt>,
    *       sera celui d'un <tt>Ver</tt> dit <i>Parfait</i>
    *   <li>false : l'<tt>Adn</tt>, contenu dans le <tt>Genotype</tt>, 
    *       sera celui d'un <tt>Ver</tt> quelconque
    * </ul>
    * @param tn la <tt>TableNourriture</tt>
    */
  public Ver(PisteGraphique piste, Point tete, String cas, int taille, boolean parfait, TableNourriture tn) 
  {
    
    // Initialisation des attributs
    this.position = new Vector(taille);
    this.genotype = new Genotype(taille, cas);
    this.vie = VIE_PAR_DEFAUT;
    this.tableCompteur = new TableCompteur(cas);
    if(tn != null)
    {
      if(parfait)
        this.genotype.getAdn().rendreAdnParfait(tn);
    }
    if(tete != null)
      position.add(tete);
    creerCorpsVer(piste);
  }
  
  /**
    * Methode qui modifie la taille du <tt>Ver</tt> <b>mais</b> ne place pas celui-ci sur la piste. <p>
    *
    * @param c la nouvelle taille du <tt>Ver</tt>
    */
  public void setTaille(int t)
  {
    genotype.setTailleVer(t);
  }
  
  /** 
    * Constructeur d'un objet <tt>Ver</tt> <b>sans l'ajouter</b>  la piste. <p>
    *
    * @param cas le cas etudie
    * @param taille la taille du <tt>Ver</tt>
    */
  public Ver(String cas, int taille) 
  {
    
    // Initialisation des attributs
    this.position = new Vector(taille);
    this.genotype = new Genotype(taille, cas);
    this.vie = VIE_PAR_DEFAUT;
    this.tableCompteur = new TableCompteur(cas);
  }
  
  /**
    * Constructeur d'un objet <tt>Ver</tt>. <p>
    *
    * <b>ATTENTION</b> : le <tt>Ver</tt> est directement ajoute a la
    * piste. <p>
    *
    * @param piste la <tt>PisteGraphique</tt> auquel il appartient
    * @param cas le cas etudie
    * @param taille la taille du <tt>Ver</tt>
    * @param typeHalo le type de halo du <tt>Ver</tt>
    *
    * @see fr.lifl.Aquarium.Simulation.Genotype
    * @see fr.lifl.Aquarium.Simulation.Adn
    */
  public Ver(PisteGraphique piste, String cas, int taille, char typeHalo) 
  {
    this(piste, null, cas, taille, typeHalo, false, null);
  }
  
  /**
    * Methode qui retourne une copie de l'objet. <p>
    *
    * @return une copie de l'objet.
    */
  public Object clone()
  {
    Ver resu = new Ver();
    resu.position = (Vector)this.position.clone();
    resu.vie = this.vie;
    resu.nbGeneration = this.nbGeneration;
    resu.genotype = (Genotype)this.genotype.clone();
    resu.direction = this.direction;
    resu.tableCompteur = (TableCompteur)this.tableCompteur.clone();
    return resu;
  }
  
  /**
    * Constructeur d'un objet <tt>Ver</tt>. <p>
    *
    * <b>ATTENTION</b> : le <tt>Ver</tt> est directement ajoute a la
    * piste. <p>
    *
    * @param piste la <tt>PisteGraphique</tt> auquel il appartient  
    * @param tete la coordonnee de sa tete
    * @param cas le cas etudie
    * @param taille la taille du <tt>Ver</tt>
    * @param typeHalo le type de halo du <tt>Ver</tt>  
    * @param parfait
    * <ul>
    *   <li>true : l'<tt>Adn</tt>, contenu dans le <tt>Genotype</tt>,
    *       sera celui d'un <tt>Ver</tt> dit <i>Parfait</i>
    *   <li>false : l'<tt>Adn</tt>, contenu dans le <tt>Genotype</tt>, 
    *       sera celui d'un <tt>Ver</tt> quelconque
    * </ul>
    * @param tn la <tt>TableNourriture</tt>
    */
  public Ver(PisteGraphique piste, Point tete, String cas, int taille, char typeHalo, boolean parfait, TableNourriture tn) 
  {
    
    // Initialisation des attributs
    this.position = new Vector(taille);
    this.genotype = new Genotype(taille, typeHalo, cas);
    this.vie = VIE_PAR_DEFAUT;
    this.tableCompteur = new TableCompteur(cas);
    if(tn != null)
    {
      if(parfait)
        this.genotype.getAdn().rendreAdnParfait(tn);
    }
    if(tete != null)
      position.add(tete);
    creerCorpsVer(piste);
  }
  
  /**
    * Methode qui modifie la taille du <tt>Ver</tt> et place celui-ci sur la piste. <p>
    *
    * @param c la nouvelle taille du <tt>Ver</tt>
    */
  public void setTaille(int t, PisteGraphique piste)
  {
    genotype.setTailleVer(t);
    
    // !!!!!!! Non teste !!!!!!!!
    
    // Permet de modifier la taille du Ver une fois celui ci construit
    if(position != null)
    {
      if(t >= position.size())
      {
        int dif = t - position.size();
        int i = t;
        while(i != 0)
        {
          Point q = getQueue();
          if(piste.caseEstTraversable(q.x + 1, q.y))
          {
            position.addElement(new Point(q.x + 1, q.y));
            i--;
            continue;
          }
          if(piste.caseEstTraversable(q.x, q.y + 1))
          {
            position.addElement(new Point(q.x + 1, q.y + 1));
            i--;
            continue;
          }
          if(piste.caseEstTraversable(q.x, q.y - 1))
          {
            position.addElement(new Point(q.x, q.y - 1));
            i--;
            continue;
          }
          if(piste.caseEstTraversable(q.x - 1, q.y))
          {
            position.addElement(new Point(q.x - 1, q.y));
            i--;
            continue;
          }
        }
        return ;
      }
      if(t < position.size())
      {
        int dif = position.size() - t;
        while(dif != 0)
        {
          Point q = getQueue();
          piste.setCaseVide(q);
          position.remove(getQueue());
          dif--;
        }
      }
    }
  }
  
  /**
    * Constructeur d'un objet <tt>Ver</tt>. <p>
    *
    * <b>ATTENTION</b> : le <tt>Ver</tt> est directement ajoute a la
    * piste. <p>
    *
    * @param piste la <tt>PisteGraphique</tt> auquel il appartient  
    * @param tete la coordonnee de sa tete
    * @param cas le cas etudie
    * @param taille la taille du <tt>Ver</tt>
    *
    * @see fr.lifl.Aquarium.Simulation.Genotype
    * @see fr.lifl.Aquarium.Simulation.Adn
    */
  public Ver(PisteGraphique piste, Point tete, String cas, int taille) 
  {
    this(piste, tete, cas, taille, false, null);
  }
  
  /**
    * Methode qui retourne la couleur du <tt>Ver</tt>. <p>
    *
    * @return la couleur du <tt>Ver</tt>
    */
  public final Color getColor()
  {
    return genotype.getCouleur();
  }
  
  /**
    * Methode qui incremente la generation du <tt>Ver</tt>. <p>
    */
  public final void incGeneration()
  {
    nbGeneration++;
  }
  
  /**
    * Methode qui modifie la couleur du <tt>Ver</tt>. <p>
    *
    * @param c la nouvelle couleur du <tt>Ver</tt>
    */
  public final void setColor(Color c)
  {
    if(genotype != null)
      genotype.setCouleur(c);
  }
  
  /**
    * Methode qui modifie la quantite de vie du <tt>Ver</tt>. <p>
    *
    * @param nbPoints la nouvelle quantite de vie du <tt>Ver</tt>
    */
  public final void setVie(float nbPoints)
  {
    vie = nbPoints;
  }
  
  /**
    * Methode qui retourne la coordonnee de la queue du <tt>Ver</tt>. <p>
    *
    * @return la coordonnee de la queue du <tt>Ver</tt>
    */
  public final Point getQueue()
  {
    Point p = null;
    try
    {
      p = (Point)position.lastElement();
    }
    catch(Exception e)
    {
      return null;
    }
    return p;
  }
  
  /**
    * Methode qui retourne la direction du <tt>Ver</tt>. <p>
    *
    * @return la direction du <tt>Ver</tt>
    */
  public final char getDirection()
  {
    return direction;
  }
  
  /**
    * Methode qui retourne la quantite de vie du <tt>Ver</tt>. <p>
    *
    * @return la quantite de vie du <tt>Ver</tt>
    */
  public final float getVie()
  {
    return vie;
  }
  
  /**
    * Methode qui retourne la taille du <tt>Ver</tt>. <p>
    *
    * @return la taille du <tt>Ver</tt>
    */
  public final int getTaille()
  {
    return genotype.getTailleVer();
  }
  
  /**
    * Methode qui modifie la direction du <tt>Ver</tt>. <p>
    *
    * @param c la nouvelle direction du <tt>Ver</tt>
    */
  public final void setDirection(char c)
  {
    direction = c;
  }
  
  /**
    * Methode qui retourne le vecteur position du <tt>Ver</tt>. <p>
    *
    * @return le vecteur position du <tt>Ver</tt>
    */
  public final Vector getPosition()
  {
    return position;
  }
  
  /**
    * Methode qui modifie le <tt>Genotype</tt> du <tt>Ver</tt>. <p>
    *
    * <b>ATTENTION !</b> ce n'est pas une copie. <p>
    *
    * @param g le nouveau <tt>Genotype</tt> du <tt>Ver</tt>
    */
  public final void setGenotype(Genotype g)
  {
    genotype = g;
  }
  
  /**
    * Methode qui met a jour la quantite de vie du <tt>Ver</tt>. <p>
    *
    * @param nbPoints la quantite de vie a <i>ajouter</i> a celle du <tt>Ver</tt>
    */
  public final void updateVie(float nbPoints)
  {
    
    /*if (genotype.haloEstProportionnel())
    System.out.println("PROP");
    else
    System.out.println("FIXE");
    System.out.println("nbPoints=" + nbPoints);*/
    vie += nbPoints;
  }
  
  /**
    * Methode qui modifie le vecteur position du <tt>Ver</tt>. <p>
    *
    * <b>ATTENTION</b> : Handle with care, ne pas oublier de retirer l'ancien corps de la piste, et/ou
    * de dessiner le nouveau. <p>
    *
    * @param v le nouveau vecteur position du <tt>Ver</tt>
    */
  public final void setPosition(Vector v)
  {
    if(position != null)
      position.removeAllElements();
    position = v;
    setTaille(v.size());
  }
  
  /**
    * Methode qui retourne la coordonnee de la tete du <tt>Ver</tt>. <p>
    *
    * @return la coordonnee de la tete du <tt>Ver</tt>
    */
  public final Point getTete()
  {
    Point p = null;
    try
    {
      p = (Point)position.firstElement();
    }
    catch(Exception e)
    {
      return null;
    }
    return p;
  }
  
  /**
    * Methode qui retourne le <tt>Genotype</tt> du <tt>Ver</tt>. <p>
    *
    * @return le <tt>Genotype</tt> du <tt>Ver</tt>
    */
  public final Genotype getGenotype()
  {
    return genotype;
  }
  
  /**
    * Methode qui m'indique ce que le <tt>Ver</tt> voit sur sa gauche
    * avec un halo fixe. <p>
    *
    * @param deuxFoisDiag
    *	<ul>
    *		<li>si faux on ne compte qu'une seule fois les cases de la diagonale (quand il regarde en face) 
    *		<li>si vrai, on compte 2 fois les cases de la diagonale (en face + a droite et a gauche).
    *	</ul>
    * @param piste la piste
    *
    * @return un caratere symbolisant ce qu'il a vu (I,N,P,M....)
    */
  protected String voirGauche(boolean deuxFoisDiag, Piste piste)
  {
    int i, j;
    int tailleHalo = genotype.getTailleHalo();
    Point tete = getTete();
    tableCompteur.initAll();
    switch(direction)
    {
      case FACE:
        for(i = 1;i <= tailleHalo;i++)
          for(j = 0;j < i;j++)
            compterCas(tete.x - i, tete.y - j, piste);
        if(deuxFoisDiag)
        {
          
          // On compte les cases de la diagonale :
          for(i = 1;i <= tailleHalo;i++)
            compterCas(tete.x - i, tete.y - i, piste);
        }
        break;
      
      case DROITE:
        for(i = 1;i <= tailleHalo;i++)
          for(j = 0;j < i;j++)
            compterCas(tete.x + j, tete.y - i, piste);
        if(deuxFoisDiag)
        {
          
          // On compte les cases de la diagonale :
          for(i = 1;i <= tailleHalo;i++)
            compterCas(tete.x + i, tete.y - i, piste);
        }
        break;
      
      case BAS:
        for(i = 1;i <= tailleHalo;i++)
          for(j = 0;j < i;j++)
            compterCas(tete.x + i, tete.y + j, piste);
        if(deuxFoisDiag)
        {
          
          // On compte les cases de la diagonale :
          for(i = 1;i <= tailleHalo;i++)
            compterCas(tete.x + i, tete.y + i, piste);
        }
        break;
      
      case GAUCHE:
        for(i = 1;i <= tailleHalo;i++)
          for(j = 0;j < i;j++)
            compterCas(tete.x - j, tete.y + i, piste);
        if(deuxFoisDiag)
        {
          
          // On compte les cases de la diagonale :
          for(i = 1;i <= tailleHalo;i++)
            compterCas(tete.x - i, tete.y + i, piste);
        }
        break;
    }
    if(tableCompteur.tousAZeroSaufI())
      return "I";
    else
      return tableCompteur.getMax();
  }
  
  /**
    * Methode qui m'indique ce que le <tt>Ver</tt> voit sur sa droite 
    * avec un halo fixe. <p>
    *
    * VISION POUR UN HALO FIXE. <p>
    *
    * @param deuxFoisDiag
    *	<ul>
    *		<li>si faux on ne compte qu'une seule fois les cases de la diagonale (quand il regarde en face)
    *		<li>si vrai, on compte 2 fois les cases de la diagonale (en face + a droite et a gauche).
    * @param piste la piste
    *
    * @return un caratere symbolisant ce qu'il a vu (I,N,P,M....)
    */
  protected String voirDroite(boolean deuxFoisDiag, Piste piste)
  {
    int i, j;
    int tailleHalo = genotype.getTailleHalo();
    Point tete = getTete();
    tableCompteur.initAll();
    switch(direction)
    {
      case FACE:
        for(i = 1;i <= tailleHalo;i++)
          for(j = 0;j < i;j++)
            compterCas(tete.x + i, tete.y - j, piste);
        if(deuxFoisDiag)
        {
          
          // On compte les cases de la diagonale :
          for(i = 1;i <= tailleHalo;i++)
            compterCas(tete.x + i, tete.y - i, piste);
        }
        break;
      
      case DROITE:
        for(i = 1;i <= tailleHalo;i++)
          for(j = 0;j < i;j++)
            compterCas(tete.x + j, tete.y + i, piste);
        if(deuxFoisDiag)
        {
          
          // On compte les cases de la diagonale :
          for(i = 1;i <= tailleHalo;i++)
            compterCas(tete.x + i, tete.y + i, piste);
        }
        break;
      
      case BAS:
        for(i = 1;i <= tailleHalo;i++)
          for(j = 0;j < i;j++)
            compterCas(tete.x - i, tete.y + j, piste);
        if(deuxFoisDiag)
        {
          
          // On compte les cases de la diagonale :
          for(i = 1;i <= tailleHalo;i++)
            compterCas(tete.x - i, tete.y + i, piste);
        }
        break;
      
      case GAUCHE:
        for(i = 1;i <= tailleHalo;i++)
          for(j = 0;j < i;j++)
            compterCas(tete.x - j, tete.y - i, piste);
        if(deuxFoisDiag)
        {
          
          // On compte les cases de la diagonale :
          for(i = 1;i <= tailleHalo;i++)
            compterCas(tete.x - i, tete.y - i, piste);
        }
        break;
    }
    if(tableCompteur.tousAZeroSaufI())
      return "I";
    else
      return tableCompteur.getMax();
  }
  
  /**
    * Methode qui cherche dans quelle direction le <tt>Ver</tt> peut
    * regarder. <p>
    *
    * Typiquement, si le <tt>Ver</tt> a une taille > 1, la direction ou
    * celui-ci regarde dependra de l'alignement entre sa tete
    * et la seconde partie de son corps. <p>
    *
    * En clair, le <tt>Ver</tt> regarde toujours dans la direction
    * qui est formee par l'alignement de sa tete et de son second
    * element.
    */
  protected char trouverDirection()
  {
    if(position == null)
      return tirerDirection();
    if(position.size() < 2)
      return tirerDirection();
    else
    {
      Point tete = getTete();
      Point p = (Point)position.elementAt(1);
      if(p.x == tete.x)
      {
        if(p.y > tete.y)
          return FACE;
        else
          return BAS;
      }
      if(p.y == tete.y)
      {
        if(p.x > tete.x)
          return GAUCHE;
        else
          return DROITE;
      }
    }
    return tirerDirection();
  }
  
  /**
    * Methode qui compte ce que le <tt>Ver</tt> mange. <p>
    *
    * @param p la case "a" mange
    * @param piste la piste
    * @param tn la <tt>TableNourriture</tt>
    */
  protected void manger(Point p, TableNourriture tn, Piste piste)
  {
    manger(p.x, p.y, tn, piste);
  }
  
  /**
    * Methode qui tire une direction au hasard. <p>
    *
    * @return une direction
    */
  protected char tirerDirection()
  {
    int i = Generateur.tirerIntEntre(4);
    switch(i)
    {
      case 0:
        return DROITE;
      
      case 1:
        return GAUCHE;
      
      case 2:
        return FACE;
      
      case 3:
        return BAS;
    }
    return DROITE;
  }
  
  /**
    * Methode qui m'indique ce que le <tt>Ver</tt> voit en face de lui. 
    * avec un halo fixe. <p>
    *
    * @param piste la piste
    *
    * @return un caratere symbolisant ce qu'il a vu (I,N,P,M....)
    */
  protected String voirFace(Piste piste)
  {
    int i, j;
    int tailleHalo = genotype.getTailleHalo();
    Point tete = getTete();
    tableCompteur.initAll();
    switch(direction)
    {
      case FACE:
        
        // colonne de face
        for(i = 1;i <= tailleHalo;i++)
          compterCas(tete.x, tete.y - i, piste);
        for(i = 1;i <= tailleHalo;i++)
          for(j = i;j <= tailleHalo;j++)
          {
            
            // partie droite
            compterCas(tete.x + i, tete.y - j, piste);
            
            // partie gauche
            compterCas(tete.x - i, tete.y - j, piste);
          }
        break;
      
      case DROITE:
        
        // colonne de face
        for(i = 1;i <= tailleHalo;i++)
          compterCas(tete.x + i, tete.y, piste);
        for(i = 1;i <= tailleHalo;i++)
          for(j = i;j <= tailleHalo;j++)
          {
            
            // partie droite
            compterCas(tete.x + j, tete.y + i, piste);
            
            // partie gauche
            compterCas(tete.x + j, tete.y - i, piste);
          }
        break;
      
      case BAS:
        
        // colonne de face
        for(i = 1;i <= tailleHalo;i++)
          compterCas(tete.x, tete.y + i, piste);
        for(i = 1;i <= tailleHalo;i++)
          for(j = i;j <= tailleHalo;j++)
          {
            
            // partie droite
            compterCas(tete.x + i, tete.y + j, piste);
            
            // partie gauche
            compterCas(tete.x - i, tete.y + j, piste);
          }
        break;
      
      case GAUCHE:
        
        // colonne de face
        for(i = 1;i <= tailleHalo;i++)
          compterCas(tete.x - i, tete.y, piste);
        for(i = 1;i <= tailleHalo;i++)
          for(j = i;j <= tailleHalo;j++)
          {
            
            // partie droite
            compterCas(tete.x - j, tete.y + i, piste);
            
            // partie gauche
            compterCas(tete.x - j, tete.y - i, piste);
          }
        break;
    }
    if(tableCompteur.tousAZeroSaufI())
      return "I";
    else
      return tableCompteur.getMax();
  }
  
  /**
    * Methode qui place le <tt>Ver</tt> sur la <tt>Piste</tt>. <p>
    *
    * Un corps de <tt>Ver</tt> ecrasera la nourriture, si il n'a pas la place. <p>
    *
    * <b>ATTENTION !</b>
    * <ul>
    *   Il est possible de boucler indefiniement ici, par exemple
    *   si je prend une <tt>Piste</tt> de 25*25 et que je veux placer
    *   5 <tt>Ver</tt> de taille 5, j'aurais des problemes !
    * </ul>
    * @param piste la piste
    */
  protected void creerCorpsVer(PisteGraphique piste)
  {
    int x = 0, y = 0, k = 1;
    char sens;
    boolean trouve = false;
    Point tete = null;
    if(position.size() != 0)
      tete = getTete();
    if(tete == null)
    {
      while(!trouve)
      {
        x = Generateur.tirerIntEntre(piste.getTailleX());
        y = Generateur.tirerIntEntre(piste.getTailleY());
        trouve = piste.caseEstTraversable(x, y);
      }
      tete = new Point(x, y);
      position.add(tete); // tete inseree
    }
    piste.setCaseVer(tete.x, tete.y, genotype.getCouleur());
    direction = tirerDirection();
    sens = direction;
    int bloquer = 0;
    while((k != getTaille()) && (k != 0))
    {
      Point queue = getQueue();
      switch(sens)
      {
        case DROITE:
          if(piste.caseEstTraversable(queue.x + 1, queue.y))
          {
            position.add(new Point(queue.x + 1, queue.y));
            piste.setCaseVer(queue.x + 1, queue.y, genotype.getCouleur());
            k++;
          }
          else
          {
            
            // Si bloquer = 4, c'est que j'ai fait un tour, je
            
            // suis donc bloque.
            if(bloquer >= 4)
            {
              retirerNouvelleTete(piste);
              return ;
            }
            sens = GAUCHE;
            bloquer++;
          }
          break;
        
        case GAUCHE:
          if(piste.caseEstTraversable(queue.x - 1, queue.y))
          {
            position.add(new Point(queue.x - 1, queue.y));
            piste.setCaseVer(queue.x - 1, queue.y, genotype.getCouleur());
            k++;
          }
          else
          {
            
            // Si bloquer = 4, c'est que j'ai fait un tour, je
            
            // suis donc bloque.
            if(bloquer >= 4)
            {
              retirerNouvelleTete(piste);
              return ;
            }
            sens = FACE;
            bloquer++;
          }
          break;
        
        case FACE:
          if(piste.caseEstTraversable(queue.x, queue.y - 1))
          {
            position.add(new Point(queue.x, queue.y - 1));
            piste.setCaseVer(queue.x, queue.y - 1, genotype.getCouleur());
            k++;
          }
          else
          {
            
            // Si bloquer = 4, c'est que j'ai fait un tour, je
            
            // suis donc bloque.
            if(bloquer >= 4)
            {
              retirerNouvelleTete(piste);
              return ;
            }
            sens = BAS;
            bloquer++;
          }
          break;
        
        case BAS:
          if(piste.caseEstTraversable(queue.x, queue.y + 1))
          {
            position.add(new Point(queue.x, queue.y + 1));
            piste.setCaseVer(queue.x, queue.y + 1, genotype.getCouleur());
            k++;
          }
          else
          {
            
            // Si bloquer = 4, c'est que j'ai fait un tour, je
            
            // suis donc bloque.
            if(bloquer >= 4)
            {
              retirerNouvelleTete(piste);
              return ;
            }
            sens = DROITE;
            bloquer++;
          }
          break;
      }
    } //fin boucle
    direction = trouverDirection();
  }
  
  /**
    * Methode qui compte ce que le <tt>Ver</tt> mange. <p>
    *
    * @param x coordonnee en X de la case "a" mange
    * @param y coordonnee en Y de la case "a" mange
    * @param piste la piste
    * @param tn la <tt>TableNourriture</tt>
    */
  protected void manger(int x, int y, TableNourriture tn, Piste piste)
  {
    char c = piste.getCase(x, y);
    float f = tn.getValeurNourriture(c);
    updateVie(f);
  }
  
  /**
    * Cette methode deplace le <tt>Ver</tt>. <p>
    *
    * Elle teste si la case (x,y) est traversable, si c'est le cas, il bouge,
    * mange si besoin est, et met a la jour la <tt>PisteGraphique</tt>. <p>
    *
    * @param p la case ou l'on desir aller
    * @param tn la <tt>TableNourriture</tt>
    * @param cout le cout d'un deplacement
    * @param piste la piste graphique pour le deplacement
    *
    * @return
    * <ul>
    *   <li>true si le <tt>Ver</tt> peut bouger
    *   <li>false sinon le <tt>Ver</tt> peut bouger
    * </ul>
    */
  protected boolean bouger(Point p, TableNourriture tn, float cout, PisteGraphique piste)
  {
    return bouger(p.x, p.y, tn, cout, piste);
  }
  
  /**
    * Methode qui m'indique ce que le <tt>Ver</tt> voit sur sa droite 
    * avec un halo proportionnel. <p>
    *
    * VISION POUR UN HALO PROPORTIONNEL. <p>
    *
    * @param tailleVision taille de la vision du ver
    * @param deuxFoisDiag
    *	<ul>
    *		<li>si faux on ne compte qu'une seule fois les cases de la diagonale (quand il regarde en face)
    *		<li>si vrai, on compte 2 fois les cases de la diagonale (en face + a droite et a gauche).
    *	</ul>
    * @param piste la piste
    *
    * @return un caratere symbolisant ce qu'il a vu (I,N,P,M....)
    */
  protected String voirDroite(int tailleVision, boolean deuxFoisDiag, Piste piste)
  {
    int i, j;
    Point tete = getTete();
    tableCompteur.initAll();
    switch(direction)
    {
      case FACE:
        for(i = 0;i < tailleVision;i++)
          compterCas(tete.x + tailleVision, tete.y - i, piste);
        if(deuxFoisDiag)
          compterCas(tete.x + tailleVision, tete.y - tailleVision, piste);
        break;
      
      case DROITE:
        for(i = 0;i < tailleVision;i++)
          compterCas(tete.x + i, tete.y + tailleVision, piste);
        if(deuxFoisDiag)
          compterCas(tete.x + tailleVision, tete.y + tailleVision, piste);
        break;
      
      case BAS:
        for(i = 0;i < tailleVision;i++)
          compterCas(tete.x - tailleVision, tete.y + i, piste);
        if(deuxFoisDiag)
          compterCas(tete.x - tailleVision, tete.y + tailleVision, piste);
        break;
      
      case GAUCHE:
        for(i = 0;i < tailleVision;i++)
          compterCas(tete.x - i, tete.y - tailleVision, piste);
        if(deuxFoisDiag)
          compterCas(tete.x - tailleVision, tete.y - tailleVision, piste);
        break;
    }
    if(tableCompteur.tousAZeroSaufI())
      return "I";
    else
      return tableCompteur.getMax();
  }
  
  /**
    * Cette methode deplace le <tt>Ver</tt>. <p>
    *
    * Elle teste si la case (x,y) est traversable, si c'est le cas, il bouge,
    * mange si besoin est, et met a la jour la <tt>PisteGraphique</tt>. <p>
    *
    * @param x coordonnee en X de la case ou l'on desir aller
    * @param y coordonnee en Y de la case ou l'on desir aller
    * @param tn la <tt>TableNourriture</tt>
    * @param cout le cout d'un deplacement
    * @param piste la piste graphique pour le deplacement
    *
    * @return
    * <ul>
    *   <li>true si le <tt>Ver</tt> peut bouger
    *   <li>false sinon le <tt>Ver</tt> peut bouger
    * </ul>
    */
  protected boolean bouger(int x, int y, TableNourriture tn, float cout, PisteGraphique piste)
  {
    if(piste.caseEstTraversable(x, y))
    {
      Point newTete = new Point(x, y);
      position.add(0, newTete);
      piste.setCaseVide(getQueue());
      position.remove(getQueue());
      if(piste.caseEstNourriture(x, y))
        manger(x, y, tn, piste.getPiste());
      piste.setCaseVer(newTete, genotype.getCouleur());
      updateVie(-cout);
      return true;
    }
    return false;
  }
  
  /**
    * Methode qui m'indique ce que le <tt>Ver</tt> voit sur sa gauche 
    * avec un halo proportionnel. <p>
    *
    * @param tailleVision taille de la vision du ver
    * @param deuxFoisDiag
    *	<ul>
    *		<li>si faux on ne compte qu'une seule fois les cases de la diagonale (quand il regarde en face) 
    *		<li>si vrai, on compte 2 fois les cases de la diagonale (en face + a droite et a gauche).
    *	</ul>
    * @param piste la piste
    *
    * @return un caratere symbolisant ce qu'il a vu (I,N,P,M....)
    */
  protected String voirGauche(int tailleVision, boolean deuxFoisDiag, Piste piste)
  {
    int i, j;
    Point tete = getTete();
    tableCompteur.initAll();
    switch(direction)
    {
      case FACE:
        for(i = 0;i < tailleVision;i++)
          compterCas(tete.x - tailleVision, tete.y - i, piste);
        if(deuxFoisDiag)
          compterCas(tete.x - tailleVision, tete.y - tailleVision, piste);
        break;
      
      case DROITE:
        for(i = 0;i < tailleVision;i++)
          compterCas(tete.x + i, tete.y - tailleVision, piste);
        if(deuxFoisDiag)
          compterCas(tete.x + tailleVision, tete.y - tailleVision, piste);
        break;
      
      case BAS:
        for(i = 0;i < tailleVision;i++)
          compterCas(tete.x + tailleVision, tete.y + i, piste);
        if(deuxFoisDiag)
          compterCas(tete.x + tailleVision, tete.y + tailleVision, piste);
        break;
      
      case GAUCHE:
        for(i = 0;i < tailleVision;i++)
          compterCas(tete.x - i, tete.y + tailleVision, piste);
        if(deuxFoisDiag)
          compterCas(tete.x - tailleVision, tete.y + tailleVision, piste);
        break;
    }
    if(tableCompteur.tousAZeroSaufI())
      return "I";
    else
      return tableCompteur.getMax();
  }
  
  /**
    * Constructeur sans parametre de <tt>Ver</tt>. <p>
    *
    * Ce constructeur ne <b><u>doit rien faire</u></b>, il sert uniquement dans
    * la methode <tt>clone()</tt>. <p>
    */
  protected Ver() 
  {
    ;
  }
  
  /**
    * Methode qui m'indique ce que le <tt>Ver</tt> voit en face de lui 
    * avec un halo proportionnel. <p>
    *
    * @param tailleVision taille de la vision du ver
    * @param piste la piste
    *
    * @return un caratere symbolisant ce qu'il a vu (I,N,P,M....)
    */
  protected String voirFace(int tailleVision, Piste piste)
  {
    int i, j;
    Point tete = getTete();
    tableCompteur.initAll();
    switch(direction)
    {
      case FACE:
        
        // colonne de face
        compterCas(tete.x, tete.y - tailleVision, piste);
        for(i = 1;i <= tailleVision;i++)
        {
          
          // partie droite
          compterCas(tete.x + i, tete.y - tailleVision, piste);
          
          // partie gauche
          compterCas(tete.x - i, tete.y - tailleVision, piste);
        }
        break;
      
      case DROITE:
        
        // colonne de face
        compterCas(tete.x + tailleVision, tete.y, piste);
        for(i = 1;i <= tailleVision;i++)
        {
          
          // partie droite
          compterCas(tete.x + tailleVision, tete.y + i, piste);
          
          // partie gauche
          compterCas(tete.x + tailleVision, tete.y - i, piste);
        }
        break;
      
      case BAS:
        
        // colonne de face
        compterCas(tete.x, tete.y + tailleVision, piste);
        for(i = 1;i <= tailleVision;i++)
        {
          
          // partie droite
          compterCas(tete.x + i, tete.y + tailleVision, piste);
          
          // partie gauche
          compterCas(tete.x - i, tete.y + tailleVision, piste);
        }
        break;
      
      case GAUCHE:
        
        // colonne de face
        compterCas(tete.x - tailleVision, tete.y, piste);
        for(i = 1;i <= tailleVision;i++)
        {
          
          // partie droite
          compterCas(tete.x - tailleVision, tete.y + i, piste);
          
          // partie gauche
          compterCas(tete.x - tailleVision, tete.y - i, piste);
        }
        break;
    }
    if(tableCompteur.tousAZeroSaufI())
      return "I";
    else
      return tableCompteur.getMax();
  }
  
  /**
    * Methode qui m'indique ce que le <tt>Ver</tt> voit. <p>
    *
    * Les premires cases que voit le ver sont celles qui sont directement
    *  sa droite,  sa gauche et en face de lui. <p>
    * Si le ver a un halo fixe, il regarde globalement  sa gauche,  sa droite
    * et en face. <p>
    *	Par contre, si son halo est proportionnel, il regarde d'abord une case
    * devant lui, s'il trouve de la nourriture il arrte de regarder, sinon il 
    * regarde un cran plus loin et ainsi de suite jusqu' trouver de la nourriture ou 
    * arriver  la taille maximale de son halo. <p>
    *
    * @param deuxFoisDiag indique si les cases diagonales doivent etre comptees deux fois ou non.
    * @param tn le table des nourritures
    * @param piste la piste
    * @param coeffAjustement la vie du ver diminuera de coeffAjustement*tailleVision
    *
    * @return une <tt>String</tt> symbolisant ce qu'il voit (PIN,III...)
    */
  protected String vision(boolean deuxFoisDiag, TableNourriture tn, Piste piste, float coeffAjustement)
  {
    int tailleHalo = genotype.getTailleHalo();
    StringBuffer cas = null;
    
    // cas du halo proportionnel
    if(genotype.haloEstProportionnel())
    {
      String tc = tn.listerType();
      boolean peutDecider = false;
      int i;
      int tailleVision = 1;
      
      // Tant que le ver n'a pas vu de nourriture, on augmente son champ de vision :
      while((tailleVision <= tailleHalo) && !peutDecider)
      {
        cas = new StringBuffer();
        cas.append(voirGauche(tailleVision, deuxFoisDiag, piste));
        cas.append(voirFace(tailleVision, piste));
        cas.append(voirDroite(tailleVision, deuxFoisDiag, piste));
        
        // on vrifie s'il a vu de la nourriture  gauche, en face ou  droite :
        for(i = 0;(i < 3) && (!peutDecider);i++)
          if(tc.indexOf(cas.charAt(i)) != -1)
            peutDecider = true;
        tailleVision++;
      }
      float perte = 0;
      
      // on enlve au ver une quantit d'nergie proportionnelle 
      
      // la taille qu'a eu son champ de vision
      
      //System.out.println("coeffAjustement=" + coeffAjustement);
      perte = (tailleVision - 1) * coeffAjustement;
      updateVie(-perte);
      return cas.toString();
    }
    
    // Cas du halo Fixe
    if(genotype.haloEstFixe())
    {
      cas = new StringBuffer();
      cas.append(voirGauche(deuxFoisDiag, piste));
      cas.append(voirFace(piste));
      cas.append(voirDroite(deuxFoisDiag, piste));
      
      // on enlve au ver une quantit d'nergie proportionnelle 
      
      // la taille de son halo		 	
      
      //System.out.println("coeffAjustement=" + coeffAjustement);		
      updateVie(-tailleHalo * coeffAjustement);
      return cas.toString();
    }
    return "III";
  }
  
  /**
    * Methode qui incremente les compteurs de nourriture
    * en fonction du contenu de la case p(x,y). <p>
    *
    * <b>Important</b> : Un <i>Mur</i> est compte comme un 'I'. <p>
    *
    * @param x coordonnee en X de la case
    * @param y coordonnee en Y de la case
    * @param piste la piste
    */
  private void compterCas(int x, int y, Piste piste)
  {
    if(piste.caseEstNourriture(x, y))
      tableCompteur.incCompteur(piste.getCase(x, y) + "");
    else
      tableCompteur.incCompteur("I");
  }
  
  /**
    * Methode qui incremente les compteurs de nourriture
    * en fonction du contenu de la case p. <p>
    *
    * <b>Important</b> : Un <i>Mur</i> n'est rien, meme pas un 'I'. <p>
    *
    * @param p la case
    * @param piste la piste
    */
  private void compterCas(Point p, Piste piste)
  {
    compterCas(p.x, p.y, piste);
  }
  
  /**
    * Methode qui est appele quand il est impossible de placer un <tt>Ver</tt>. <p>
    * 
    * On tue le <tt>Ver</tt>, ou du moins ce qu'il en est, et on retente
    * un placement aleatoire. <p>
    */
  private final void retirerNouvelleTete(PisteGraphique piste)
  {
    System.out.println("-----------Retirage de tete-----------");
    tuerVer(piste);
    creerCorpsVer(piste);
  }
  
  /** Sert indique si l'on doit ou non compter deux fois les cases diagonales. <p> */
  public static boolean DIAGONALE = false;
  
  /** Indique la direction et le sens BAS. <p> */
  public final static char BAS = 'B';
  
  /** Indique la direction et le sens FACE. <p> */
  public final static char FACE = 'F';
  
  /** Indique la direction et le sens GAUCHE. <p> */
  public final static char GAUCHE = 'G';
  
  /** La vie par defaut d'un <tt>Ver</tt> est de 150 pts. <p> */
  public final static int VIE_PAR_DEFAUT = 150;
  
  /** Indique la direction et le sens DROITE. <p> */
  public final static char DROITE = 'D';
  
  /** Indique a quelle generation appartient le <tt>Ver</tt>. <p> */
  protected int nbGeneration = 0;
  
  /** Le <tt>Genotype</tt> du <tt>Ver</tt>. <p> */
  protected Genotype genotype = null;
  
  /** Indique ou regarde le <tt>Ver</tt>. <p> */
  protected char direction;
  
  /** Permet de compter le nbr de nourritures vue, a droite,
    * a gauche, en face. <p>
    */
  protected TableCompteur tableCompteur = null;
  
  /** Contient les coordonnees du <tt>Ver</tt>. <p>
    *
    * Celles-ci sont rangees par ordre, c.a.d, en place 0 on
    * trouve la tete du <tt>Ver</tt>. <p>
    */
  protected Vector position = null;
  
  /** La quantite de vie que possede le <tt>Ver</tt>. <p> */
  protected float vie = 0;
}
