package fr.lifl.Aquarium.Simulation;
import java.awt.Color;
import java.util.Hashtable;
import java.util.Enumeration;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

/**
  * Cette <i>classe</i> represente tous les types de nourritures 
  * qui seront disponibles durant la simulation. <p>
  *
  * Derniere modification 22 Janvier 2000 <br>
  *
  * @author
  * <ul>
  *   <li><a href="mailto:renaud91@hotmail.com">Ferret Renaud</a> (V1.++)
  * </ul>
  *
  * @see fr.lifl.Aquarium.Simulation.Nourriture
  *
  * @version 1.2
  */
public class TableNourriture implements Serializable,Cloneable
{
  
  /**
    * Surcharge de toString(). <p>
    *
    * Exemple : <br>
    * [ [E,0.6131362,lightgray],[D,0.70678157,green],
    * [C,0.5156116,gray],[B,0.5009792,darkgray],[A,0.851057,cyan] ]. <p>
    *
    * @return une chaine de caracteres contenant toutes les Nourritures de
    * la table.
    */
  public String toString()
  {
    int i;
    Nourriture tabN[] = lister();
    if(tabN == null)
      return "[null]";
    StringBuffer resu = new StringBuffer();
    resu.append("[ ");
    for(i = 0;i < tabN.length - 1;i++)
    {
      resu.append(tabN[i]);
      resu.append(", ");
    }
    resu.append(tabN[i]);
    resu.append(" ]");
    return String.valueOf(resu);
  }
  
  /**
    * Methode qui sauve une Table de Nourriture. <P>
    *
    * @param nomFichier le nom du fichier ou sauver la Table de Nourriture.
    *
    * @return
    * <ul>
    *   <li>true si tout c'est bien passe
    *   <li>false sinon
    * </ul>
    */
  public boolean sauver(String nomFichier)
  {
    int i = 0;
    if(nomFichier == null)
      return false;
    if(nomFichier.trim().equals(""))
      return false;
    try
    {
      FileOutputStream fichierOuvert = new FileOutputStream(nomFichier);
      ObjectOutputStream out = new ObjectOutputStream(fichierOuvert);
      out.writeObject(this);
      out.flush();
      out.close();
      fichierOuvert.close();
    }
    catch(Exception e)
    {
      System.err.println(Erreur.getString("erreur_tablenourriture_sauver"));
      return false;
    }
    return true;
  }
  
  /**
    * Retire une nourriture de la table. <p>
    *
    * @param n la Nourriture a retirer.
    *
    * @return
    * <ul>
    *   <li> true si tout c'est bien passe
    *   <li> false sinon
    * </ul>
    */
  public boolean removeNourriture(Nourriture n)
  {
    if(n == null)
    {
      System.err.println(Erreur.getString("erreur_tablenourriture_removenourriture_n"));
      return false;
    }
    return removeNourriture(n.getType());
  }
  
  /**
    * Ajoute un type de nourriture a la table. <p>
    *
    * C'est a ce moment qu'une couleur est affectee a la
    * nourriture ajoutee. <p>
    *
    * La clef de la nourriture ajoutee est son type. <p>
    *
    * Le type de la Nourriture et sa valeur sont generes
    * automatiquement. </p>
    *
    * @return
    * <ul>
    *   <li> true si tout c'est bien passe
    *   <li> false sinon
    * </ul>
    */
  public boolean putNourriture()
  {
    int i = 0;
    for(i = 0;i < tableType.length;i++)
    {
      Nourriture n = new Nourriture(tableType[i]);
      if(putNourriture(n))
        return true;
    }
    return false;
  }
  
  /**
    * Ajoute un type de nourriture a la table. <p>
    *
    * C'est a ce moment qu'une couleur est affectee a la
    * nourriture ajoutee. <p>
    *
    * La clef de la nourriture ajoutee est son type. <p>
    *
    * @param n la Nourriture a ajouter.
    *
    * @return
    * <ul>
    *   <li> true si tout c'est bien passe
    *   <li> false sinon
    * </ul>
    */
  public boolean putNourriture(Nourriture n)
  {
    if(n == null)
    {
      System.err.println(Erreur.getString("erreur_tablenourriture_putnourriture_n"));
      return false;
    }
    if(indiceCouleur >= tableCouleur.length)
      indiceCouleur = 0;
    if(n.getCouleur() == null)
      n.setCouleur(tableCouleur[indiceCouleur]);
    if(!tableNourriture.containsKey(new Character(n.getType())))
    {
      tableNourriture.put(new Character(n.getType()), n);
      indiceCouleur++;
      return true;
    }
    return false;
  }
  
  /**
    * Methode qui liste le contenu de la table de nourritures. <p>
    *
    * Possibilite de ranger le tableau par ordre decroissant des
    * valeurs de nourriture. <p>
    *
    * @param parOrdreDeCroissant
    * <ul>
    *   <li>si egale true, le tableau sera range par ordre decroissant
    *       des valeurs energetiques, de la plus energetique, vers la
    *       moins energetique
    *   <li>si false, le tableau est range par ordre de type. (Z est le plus
    *       fort, et A le plus faible)
    * </ul>
    *
    * @return un tableau de Nourriture.
    */
  public Nourriture[] lister(boolean parOdreDeCroissant)
  {
    int i = 0;
    Nourriture tab[] = new Nourriture[tableNourriture.size()];
    Enumeration e = tableNourriture.keys();
    while(e.hasMoreElements())
    {
      Character c = (Character)(e.nextElement());
      Nourriture n = getNourriture(c.charValue());
      tab[i] = n;
      i++;
    }
    if(parOdreDeCroissant)
    {
      Nourriture max = (Nourriture)tab[0].clone();
      for(int j = 0;j < tab.length;j++)
        for(i = 0;i < tab.length;i++)
        {
          if(tab[j].getValeur() > tab[i].getValeur())
          {
            Nourriture tmp = (Nourriture)tab[j].clone();
            tab[j] = (Nourriture)tab[i].clone();
            tab[i] = tmp;
          }
        }
    }
    return tab;
  }
  
  /**
    * Methode qui liste le contenu de la table de nourriture. <p>
    *
    * Le tableau est range dans l'ordre des types. <p>
    *
    * @return un tableau de Nourriture.
    */
  public Nourriture[] lister()
  {
    return lister(false);
  }
  
  /**
    * Methode qui retourne une copie de l'objet. <p>
    *
    * @return une copie de l'objet.
    */
  public Object clone()
  {
    TableNourriture resu = new TableNourriture();
    resu.tableNourriture = (Hashtable)this.tableNourriture.clone();
    return resu;
  }
  
  /**
    * Constructeur d'une Table de Nourriture. <p>
    */
  public TableNourriture() 
  {
    this(0);
  }
  
  /**
    * Cette methode donne la liste des clefs, et donc des types
    * de nourritures. <p>
    *
    * <b>ATTENTION !</b>
    * <ul>
    *   N'oubliez pas "I" si vous avez besoin, car il sera <b>JAMAIS</b>
    *   dans le resultat de cette methode.
    * </ul>
    *
    * @return une chaine contenant tous les types de nourritures
    * contenus dans la table.
    */
  public String listerType()
  {
    String resu = "";
    Enumeration e = tableNourriture.keys();
    while(e.hasMoreElements())
    {
      Character c = (Character)(e.nextElement());
      resu = resu + c.charValue();
    }
    return resu;
  }
  
  /**
    * Constructeur d'une Table de Nourriture. <p>
    *
    * @param nb le nombre de nourritures a generer.
    */
  public TableNourriture(int nb) 
  {
    tableNourriture = new Hashtable();
    indiceCouleur = 0;
    if(nb > 0)
      putNourriture(nb);
  }
  
  /**
    * Ajoute un nombre de nourritures a la table. <p>
    *
    * La clef de la nourriture ajoutee est son type, et les types sont
    * deduits de la chaine de caracteres, un charactere = un type de 
    * nourriture. <p>
    *
    * Les valeurs energetiques sont generees automatiquement. <p>
    *
    * Si il y a une lettre en double (comme C dans CEARCZWI), l'ajout
    * des types de nourritures se fera correctement, sans qu'il y est 
    * deux fois C, mais la methode retournera false. <p>
    *
    * @return
    * <ul>
    *   <li> true si tout c'est bien passe
    *   <li> false sinon
    * </ul>
    */
  public boolean putNourriture(String types)
  {
    int i = 0;
    boolean resu = true;
    if(types == null)
      return false;
    if(types.trim().equals(""))
      return false;
    for(i = 0;i < types.length();i++)
      resu = resu && putNourriture(new Nourriture(types.charAt(i)));
    return resu;
  }
  
  /**
    * Constructeur d'une Table de Nourriture. <p>
    *
    * @param types une chaine de caracteres representant tous les
    * types de nourriture.
    */
  public TableNourriture(String types) 
  {
    this(0);
    putNourriture(types);
  }
  
  /**
    * Retire une nourriture de la table. <p>
    *
    * @param c le type de la <tt>Nourriture</tt> a retirer
    *
    * @return
    * <ul>
    *   <li> true si tout c'est bien passe
    *   <li> false sinon
    * </ul>
    */
  public boolean removeNourriture(char c)
  {
    if(tableNourriture.remove(new Character(c)) != null)
      return true;
    return false;
  }
  
  /**
    * Ajoute un nombre de nourritures a la table. <p>
    *
    * C'est a ce moment qu'une couleur est affectee a la
    * nourriture ajoutee. <p>
    *
    * La clef de la nourriture ajoutee est son type. <p>
    *
    * Les differents types de Nourriture et leur valeur sont 
    * generes automatiquement. </p>
    *
    * @return
    * <ul>
    *   <li> true si tout c'est bien passe
    *   <li> false sinon
    * </ul>
    */
  public boolean putNourriture(int nb)
  {
    int i = 0;
    boolean resu = true;
    if(nb == 0)
      return true;
    if(nb < 0)
      return false;
    for(i = 0;i < nb;i++)
    {
      resu = resu && putNourriture();
      if(!resu)
        return false;
    }
    return true;
  }
  
  /**
    * Methode qui retourne le nombre d'elements dans la table. <p>
    *
    * @return le nombre d'elements contenus dans la table.
    */
  public final int getSize()
  {
    return tableNourriture.size();
  }
  
  /**
    * Obtenir un objet nourriture de la table. <p>
    *
    * @param t le type de la nourriture
    *
    * @return
    * <ul>
    *   <li> l'objet Nourriture si celui-ci est dans la table
    *   <li> null sinon
    * </ul>
    */
  public final Nourriture getNourriture(char t)
  {
    Nourriture n = null;
    n = (Nourriture)tableNourriture.get(new Character(t));
    return n;
  }
  
  /**
    * Donne la nourriture la moins energetique. <p>
    *
    * @return la Nourriture la moins energetique
    */
  public final Nourriture getWorstNourriture()
  {
    if(tableNourriture == null)
      return null;
    Nourriture min = new Nourriture('!', Float.MAX_VALUE);
    Nourriture n = null;
    Character c = null;
    Enumeration e = tableNourriture.keys();
    while(e.hasMoreElements())
    {
      c = (Character)(e.nextElement());
      n = getNourriture(c.charValue());
      if(n.getValeur() < min.getValeur())
        min = n;
    }
    return min;
  }
  
  /**
    * Donne la nourriture la plus energetique. <p>
    *
    * @return la Nourriture la plus energetique
    */
  public final Nourriture getBestNourriture()
  {
    if(tableNourriture == null)
      return null;
    Nourriture max = new Nourriture('!', Float.MIN_VALUE);
    Nourriture n = null;
    Character c = null;
    Enumeration e = tableNourriture.keys();
    while(e.hasMoreElements())
    {
      c = (Character)(e.nextElement());
      n = getNourriture(c.charValue());
      if(n.getValeur() > max.getValeur())
        max = n;
    }
    return max;
  }
  
  /**
    * Obtenir la couleur d'une nourriture
    * contenue dans la table. <p>
    *
    * @param t le type de la nourriture
    *
    * @return
    * <ul>
    *   <li> la couleur de la nourriture de type t
    *   <li> null sinon
    * </ul>
    */
  public final Color getCouleurNourriture(char t)
  {
    Nourriture n = null;
    Color c = null;
    n = (Nourriture)tableNourriture.get(new Character(t));
    if(n != null)
      c = n.getCouleur();
    return c;
  }
  
  /**
    * Obtenir la valeur energetique d'une nourriture
    * contenue dans la table. <p>
    *
    * @param t le type de la nourriture
    *
    * @return
    * <ul>
    *   <li> la valeur energetique de la nourriture de type t
    *   <li> 0.0 sinon
    * </ul>
    */
  public final float getValeurNourriture(char t)
  {
    Nourriture n = null;
    float vE = 0;
    n = (Nourriture)tableNourriture.get(new Character(t));
    if(n != null)
      vE = n.getValeur();
    return vE;
  }
  
  /**
    * Charge une Table de Nourriture. <P>
    *
    * @param nomFichier le nom du fichier d'ou charger la Table de Nourriture
    *
    * @return
    * <ul>
    *   <li>la TableNourriture contenue dans le fichier
    *   <li>null sinon
    * </ul>
    */
  public static TableNourriture charger(String nomFichier)
  {
    try
    {
      FileInputStream fichierOuvert = new FileInputStream(nomFichier);
      ObjectInputStream in = new ObjectInputStream(fichierOuvert);
      TableNourriture tN = (TableNourriture)in.readObject();
      in.close();
      fichierOuvert.close();
      return tN;
    }
    catch(Exception e)
    {
      System.err.println(Erreur.getString("erreur_tablenourriture_charger"));
      return null;
    }
  }
  
  /**
    * Couleurs disponibles pour les <tt>Nourriture</tt>. <p>
    *
    * <b>ATTENTION !</b> :
    * <ul>
    *   <font color="black">Color.black : Murs</font>
    *   <font color="blue">Color.blue : aire de jeu</font>
    *   <font color="white">Color.white : Vers</font>
    * </ul>
    */
  public final static Color tableCouleur[] = 
  {
    Color.cyan, Color.green, Color.magenta, Color.orange, 
    Color.pink, Color.red, Color.yellow, Color.darkGray, 
    Color.gray, Color.lightGray
  };
  
  /**
    * Lettre disponible pour un type de nourriture. <p>
    *
    * <b>ATTENTION !</b> 
    * <ul>
    *   'M' et 'I' sont reserves
    * </ul>
    */
  public final static char tableType[] = 
  {
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L', 
    'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 
    'Y', 'Z'
  };
  
  /** Les <tt>Nourriture</tt> sont rangees dans une <tt>Hashtable</tt>. <p> */
  protected Hashtable tableNourriture = null;
  
  /** Indice courant de la couleur. <p>
    *
    * N'est utilise que lors de l'affectation automatique
    *	de la couleur pour une <tt>Nourriture</tt>. <p>
    */
  private int indiceCouleur;
}
