package fr.lifl.Aquarium.Simulation;
import java.awt.Point;
import java.awt.Panel;
import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Dimension;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import javax.swing.JComponent;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

/**
  * La <i>classe</i> <tt>PisteGraphique</tt> contient l'aire de vie. <P>
  *
  * Derniere modification 23 Janvier 2000. <p>
  *
  * @author
  * <ul>
  *   <li>Hiblot Sebastien (V1.0)
  *   <li><a href="mailto:renaud91@hotmail.com">Ferret Renaud</a> (V3.++)
  * </ul>
  *
  * @version 3.2
  *
  * @see fr.lifl.Aquarium.Simulation.Aquarium
  * @see fr.lifl.Aquarium.Simulation.Piste
  */
public class PisteGraphique extends JComponent implements Serializable,Cloneable,MouseListener
{
  
  /**
    * Methode qui modifie la taille de <tt>PisteGraphique</tt>. <P>
    *
    *	Le composant graphique anisi que la <tt>Piste</tt> sont modifies. <p> 
    *
    * @param taille la nouvelle taille de la <tt>PisteGraphique</tt>, celle ci
    *	est suposee carre.
    */
  public void setTaille(Dimension taille)
  {
    setTaille(taille.height, taille.width);
  }
  
  /**
    * Methode qui modifie la taille de <tt>PisteGraphique</tt>. <P>
    *
    *	Le composant graphique anisi que la <tt>Piste</tt> sont modifies. <p> 
    *
    * @param tailleX la nouvelle taille en X de la <tt>PisteGraphique</tt>
    * @param tailleY la nouvelle taille en Y de la <tt>PisteGraphique</tt>
    */
  public void setTaille(int tailleX, int tailleY)
  {
    piste = null;
    piste = new Piste(tailleX, tailleY);
    taillePix = new Dimension(tailleX * mult, tailleY * mult); // taille en pixels
    setPreferredSize(taillePix);
  }
  
  /**
    * Methode qui modifie la taille de <tt>PisteGraphique</tt>. <P>
    *
    *	Le composant graphique anisi que la <tt>Piste</tt> sont modifies. <p> 
    *
    * @param taille la nouvelle taille de la <tt>PisteGraphique</tt>, celle ci
    *	est suposee carre
    */
  public void setTaille(int taille)
  {
    setTaille(taille, taille);
  }
  
  /**
    * Permet de placer de la nourriture sur la piste de jeu,
    * aleatoirement. <P>
    *
    * Place un type <tt>t</tt> de nourriture de facon circulaire
    * sur la <tt>Piste</tt>. <p>
    *
    * @param x la coordonnee en X du centre
    * @param y la coordonnee en Y du centre
    * @param r le rayon du cercle
    * @param tn la <tt>TableNourriture</tt> sui contient les types de
    * nourritures a distribuer
    */
  public void placerNourriture(int x, int y, int r, TableNourriture tn)
  {
    placerNourriture(x, y, r, QUANTITE_NOURRITURE, tn);
  }
  
  /**
    * Surcharge de paint. <P>
    *
    * Ici, la methode n'est effective <b>QUE SI</b> 
    * <tt>GestionSimulation.ETAT_COURANT</tt> est different de 
    * <tt>GestionSimulation.ACCELERER</tt>. </p>
    *
    * @param g le Graphics a peindre
    */
  public synchronized void paint(Graphics g)
  {
    if(GestionSimulation.ETAT_COURANT != GestionSimulation.ACCELERER)
    {
      if(oi != null)
        g.drawImage(oi, 0, 0, this);
      else
        initGraphique();
    }
  }
  
  /**
    * Methode qui renvoie true si la case est traversable. <p>
    *
    * Une case traversable est soit de la <b>Nourriture</b>
    * soit <b>VIDE</b>. <p>
    *
    * @return
    * <ul>
    *   <li>true si la case est traversable
    *   <li>false si la case contient quelque chose de non traversable
    * </ul>
    */
  public boolean caseEstTraversable(Point p)
  {
    return piste.caseEstTraversable(p.x, p.y);
  }
  
  /**
    * Methode qui renvoie true si la case est VIDE. <p>
    *
    * @param p <tt>Point</tt> qui serait VIDE
    *
    * @return
    * <ul>
    *   <li>true si la case contient VIDE
    *   <li>false si la case contient autre chose
    * </ul>
    */
  public boolean caseEstVide(Point p)
  {
    return piste.caseEstVide(p.x, p.y);
  }
  
  /**
    * Permet de placer de la nourriture sur la piste de jeu
    * aleatoirement. <P>
    *
    * @param n la nourriture a placer
    */
  public void placerNourriture(Nourriture n)
  {
    placerNourriture(QUANTITE_NOURRITURE, n);
  }
  
  /**
    */
  public void setPiste(Piste p)
  {
    piste = p;
    taillePix = new Dimension(p.getTailleX() * mult, p.getTailleY() * mult);
    setPreferredSize(taillePix);
  }
  public void mouseEntered(MouseEvent e)
  {
    e.consume();
  }
  
  /**
    * Methode qui place le VIDE dans la case. <P>
    *
    * @param p <tt>Point</tt> a remplir
    */
  public void setCaseVide(Point p)
  {
    if(p != null)
      setCaseVide(p.x, p.y);
  }
  
  /**
    * Methode qui renvoie true si la case est VER. <p>
    *
    * @param <tt>Point</tt> qui serait un Ver
    *
    * @return
    * <ul>
    *   <li>true si la case contient VER
    *   <li>false si la case contient autre chose
    * </ul>
    */
  public boolean caseEstVer(Point p)
  {
    return piste.caseEstVer(p.x, p.y);
  }
  
  /**
    * Methode qui place le VIDE dans la case (x,y). <P>
    *
    * @param x Coordonnee en x du caractere VIDE
    * @param y Coordonnee en y du caractere VIDE
    */
  public void setCaseVide(int x, int y)
  {
    piste.setCaseVide(x, y);
    affichePoint(x, y, COULEUR_VIDE);
  }
  
  /**
    * Methode qui retourne une copie de l'objet. <p>
    *
    * @return une copie de l'objet.
    */
  public Object clone()
  {
    PisteGraphique resu = new PisteGraphique(this.getTaille());
    resu.piste = (Piste)this.piste.clone();
    return resu;
  }
  
  /**
    * Methode qui place le VER dans la case. <P>
    *
    * @param p <tt>Point</tt> a remplir
    * @param c la couleur du <tt>Ver</tt>
    */
  public void setCaseVer(Point p, Color c)
  {
    if(p != null)
      setCaseVer(p.x, p.y, c);
  }
  
  /**
    * Methode qui renvoie le caractere contenu dans la case. <p>
    *
    * @param p le <tt>Point</tt> dont on veut connaitre la valeur
    *
    * @return le caractere contenu dans la case (x,y) de la piste
    */
  public char getCase(Point p)
  {
    return piste.getCase(p.x, p.y);
  }
  
  /**
    * Methode qui place le VER dans la case (x,y). <P>
    *
    * @param x Coordonnee en x du caractere VER
    * @param y Coordonnee en y du caractere VER
    * @param c la couleur du <tt>Ver</tt>
    */
  public void setCaseVer(int x, int y, Color c)
  {
    piste.setCaseVer(x, y);
    affichePoint(x, y, c);
  }
  
  /**
    * Permet de placer de la nourriture sur la piste de jeu,
    * aleatoirement. <P>
    *
    * Place un type <tt>t</tt> de nourriture de facon circulaire
    * sur la <tt>Piste</tt>. <p>
    *
    * Tous les types de nourritures sont places, alors
    * <b>ATTENTION</b> a ne pas choisir <tt>nb</tt> trop grand. <p>
    *
    * @param x la coordonnee en X du centre
    * @param y la coordonnee en Y du centre
    * @param r la largeur du carre
    * @param nb la quantite de nourriture a placer
    * @param tn la <tt>TableNourriture</tt> sui contient les types de
    * nourritures a distribuer
    */
  public void placerNourriture(int x, int y, int r, int nb, TableNourriture tn)
  {
    if(tn != null)
    {
      int i;
      Nourriture tabN[] = tn.lister();
      for(i = 0;i < tabN.length;i++)
        placerNourriture(x, y, r, nb, tabN[i]);
    }
  }
  
  /**
    * Methode qui place le MUR dans la case. <P>
    *
    * @param p <tt>Point</tt> a remplir
    */
  public void setCaseMur(Point p)
  {
    if(p != null)
      setCaseMur(p.x, p.y);
  }
  
  /**
    * Methode qui lie une TableNourriture a la piste. <p>
    *
    * @param tn la table qui sera liee.
    */
  public void lierTableNourriture(TableNourriture tn)
  {
    tableN = tn;
  }
  
  /**
    * Methode qui place le MUR dans la case (x,y). <P>
    *
    * @param x Coordonnee en x du caractere MUR
    * @param y Coordonnee en y du caractere MUR
    */
  public void setCaseMur(int x, int y)
  {
    piste.setCaseMur(x, y);
    affichePoint(x, y, COULEUR_MUR);
  }
  public void mousePressed(MouseEvent e)
  {
    if(tableN != null)
    {
      if((TAILLE_PINCEAU == -1) && (TYPE_NOURRITURE == '*'))
        placerNourriture(QUANTITE_NOURRITURE, tableN);
      if((TAILLE_PINCEAU == -1) && (TYPE_NOURRITURE != '*'))
        placerNourriture(QUANTITE_NOURRITURE, tableN.getNourriture(TYPE_NOURRITURE));
      if(e != null)
      {
        if((TAILLE_PINCEAU != -1) && (TYPE_NOURRITURE == '*'))
          placerNourriture(e.getX() / mult, e.getY() / mult, TAILLE_PINCEAU, QUANTITE_NOURRITURE, tableN);
        if((TAILLE_PINCEAU != -1) && (TYPE_NOURRITURE != '*'))
          placerNourriture(e.getX() / mult, e.getY() / mult, TAILLE_PINCEAU, QUANTITE_NOURRITURE, tableN.getNourriture(TYPE_NOURRITURE));
      }
    }
    else
      System.err.println(Erreur.getString("erreur_pistegraphique_mousepressed"));
    if(e != null)
      e.consume();
  }
  
  /**
    * Constructeur d'un objet <tt>PisteGraphique</tt>. <P>
    *
    * Le <tt>PisteGraphique<tt> est supose <b>carre</b> (taille*taille). <p>
    *
    * @param taille la taille du <tt>PisteGraphique</tt>, en nombre de points 
    * affichables (c.a.d la taille d'une <tt>Piste</tt> par exemple)
    */
  public PisteGraphique(int taille) 
  {
    this(taille, taille);
  }
  
  /**
    * Methode qui renvoie true si la case (x,y) est VER. <p>
    *
    * @param x Coordonnee en x du caractere
    * @param y Coordonnee en y du caractere
    *
    * @return
    * <ul>
    *   <li>true si la case contient VER
    *   <li>false si la case contient autre chose
    * </ul>
    */
  public boolean caseEstVer(int x, int y)
  {
    return piste.caseEstVer(x, y);
  }
  
  /**
    * Constructeur d'un objet <tt>PisteGraphique</tt>. <P>
    *
    * @param tailleX la taille en X du <tt>PisteGraphique</tt>
    * @param tailleY la taille en Y du <tt>PisteGraphique</tt>
    */
  public PisteGraphique(int tailleX, int tailleY) 
  {
    super();
    this.addMouseListener(this);
    
    // taille en pixels
    taillePix = new Dimension(tailleX * mult, tailleY * mult);
    piste = new Piste(tailleX, tailleY);
    setPreferredSize(taillePix);
  }
  
  /**
    * Permet de placer de la nourriture sur la piste de jeu
    * aleatoirement. <P>
    *
    * @param nb la quantite de nourriture a placer
    * @param n la nourriture a placer
    */
  public void placerNourriture(int nb, Nourriture n)
  {
    int combien = nb;
    
    // Dans le but de ne pas faire une boucle infinie
    double maxTentatives = (piste.getTailleX() * piste.getTailleY()) - 1;
    int i;
    for(i = 0;((i <= nb) && (maxTentatives > 0));i++)
    {
      boolean trouve = false;
      int X = 0;
      int Y = 0;
      while(!trouve)
      {
        X = Generateur.tirerIntEntre(piste.getTailleX());
        Y = Generateur.tirerIntEntre(piste.getTailleY());
        trouve = (piste.caseEstTraversable(X, Y));
      }
      maxTentatives--;
      piste.setCase(X, Y, n.getType());
      affichePoint(X, Y, n.getCouleur());
    }
    this.repaint();
  }
  
  /**
    * Constructeur d'un objet <tt>PisteGraphique</tt>. <P>
    *
    * @param p la <tt>Piste</tt> de base.
    */
  public PisteGraphique(Piste p) 
  {
    this(p.getTailleX(), p.getTailleY());
    piste = p;
  }
  
  /**
    * Permet de placer de la nourriture sur la piste de jeu
    * aleatoirement. <P>
    *
    * @param x la coordonnee en x ou placer la nourriture
    * @param y la coordonnee en y ou placer la nourriture
    * @param n la nourriture a placer
    */
  public void placerNourriture(int x, int y, Nourriture n)
  {
    if(piste.caseEstTraversable(x, y))
    {
      piste.setCase(x, y, n.getType());
      affichePoint(x, y, n.getCouleur());
    }
  }
  
  /**
    * Constructeur d'un objet <tt>PisteGraphique</tt>. <P>
    *
    * @param taille la taille du <tt>PisteGraphique</tt>, en nombre de points 
    * affichables (c.a.d la taille d'une <tt>Piste</tt> par exemple)
    */
  public PisteGraphique(Dimension d) 
  {
    this(d.width, d.height);
  }
  
  /**
    * Methode qui fabrique un mur allant de (x1,y1) a (x2,y2). <p>
    *
    * <b>ATTENTION</b> : ne trace pas <i>encore</i> les droites en biais. <p>
    *
    * @param x1 Coordonnee en x du point de depart en haut a gauche du mur.
    * @param y1 Coordonnee en y du point de depart en haut a gauche du mur.
    * @param x2 Coordonnee en x du point de depart en bas a droite du mur.
    * @param y2 Coordonnee en y du point de depart en bas a droite du mur.
    */
  public void creerMur(int x1, int y1, int x2, int y2)
  {
    if((x1 > x2) || (y1 > y2))
      return ;
    int x, y;
    
    // cas de la droite
    
    // Sur x
    if(y1 == y2)
    {
      for(x = x1;x < x2;x++)
        setCaseMur(x, y1);
      return ;
    }
    
    // Sur y
    if(x1 == x2)
    {
      for(y = y1;y < y2;y++)
        setCaseMur(x1, y);
      return ;
    }
  }
  
  /**
    * Affichage d'une <tt>Piste</tt>. <P>
    *
    * @param tn une <tt>TableNourriture</tt> pour les couleurs des <tt>Nourriture</tt>
    * @param lV une <tt>ListeVer</tt> pour les couleurs des <tt>Ver</tt>
    */
  public void affichePiste(TableNourriture tn, ListeVer lV)
  {
    if(GestionSimulation.ETAT_COURANT != GestionSimulation.ACCELERER)
    {
      
      // Tout, sauf les Ver
      int i, j;
      clean();
      for(i = 0;i < piste.getTailleX();i++)
        for(j = 0;j < piste.getTailleY();j++)
        {
          if(piste.caseEstNourriture(i, j))
          {
            if(tn != null)
              affichePoint(i, j, tn.getCouleurNourriture(piste.getCase(i, j)));
            continue;
          }
          if(piste.caseEstMur(i, j))
          {
            affichePoint(i, j, COULEUR_MUR);
            continue;
          }
        }
      
      // Les Vers
      if(lV != null)
      {
        for(i = 0;i < lV.getSize();i++)
          afficheVer(lV.getVer(i));
      }
      repaint();
    }
  }
  
  /**
    * Methode qui renvoie le caractere contenu dans la case (x,y). <p>
    *
    * @param x Coordonnee en x du caractere
    * @param y Coordonnee en y du caractere
    *
    * @return le caractere contenu dans la case (x,y) de la piste
    */
  public char getCase(int x, int y)
  {
    return piste.getCase(x, y);
  }
  
  /**
    * Methode qui sauve une <tt>PisteGraphique</tt>. <P>
    *
    * <b>Attention !</b> : Ne fonctionne pas tres bien, <font color="red">handle with care !</font>. <p>
    *
    * @param nomFichier le nom du fichier ou sauver la piste.
    *
    * @return
    * <ul>
    *   <li>true si tout c'est bien passe
    *   <li>false sinon
    * </ul>
    */
  public boolean sauver(String nomFichier)
  {
    int i = 0;
    if(nomFichier == null)
      return false;
    if(nomFichier.trim().equals(""))
      return false;
    try
    {
      FileOutputStream fichierOuvert = new FileOutputStream(nomFichier);
      ObjectOutputStream out = new ObjectOutputStream(fichierOuvert);
      out.writeObject(this);
      out.flush();
      out.close();
      fichierOuvert.close();
    }
    catch(Exception e)
    {
      System.err.println(Erreur.getString("erreur_pistegraphique_sauver"));
      return false;
    }
    return true;
  }
  
  /**
    * Permet de placer de la nourriture sur la piste de jeu,
    * aleatoirement. <P>
    *
    * Un nombre <tt>nb</tt> de chaque type de <tt>Nourriture</tt> sera
    * place sur la <tt>Piste</tt>. <p>
    *
    * Les types de nourritures sont directements tires de la 
    * TableNourriture. <p>
    *
    * @param nb la quantite de nourriture a placer
    * @param tn la <tt>TableNourriture</tt> sui contient les types de
    * nourritures a distribuer
    */
  public void placerNourriture(int nb, TableNourriture tn)
  {
    if(tn != null)
    {
      int i;
      Nourriture tabN[] = tn.lister();
      for(i = 0;i < tabN.length;i++)
        placerNourriture(nb, tabN[i]);
    }
  }
  
  /**
    * Permet de placer de la nourriture sur la piste de jeu,
    * aleatoirement. <P>
    *
    * Les types de nourritures sont directements tires de la 
    * TableNourriture. <p>
    *
    * @param tn la <tt>TableNourriture</tt> sui contient les types de
    * nourritures a distribuer
    */
  public void placerNourriture(TableNourriture tn)
  {
    placerNourriture(QUANTITE_NOURRITURE, tn);
  }
  
  /**
    * Permet de placer de la nourriture sur la piste de jeu,
    * aleatoirement. <P>
    *
    * Place un type <tt>t</tt> de nourriture de facon circulaire
    * sur la <tt>Piste</tt>. <p>
    *
    * @param x la coordonnee en X du centre
    * @param y la coordonnee en Y du centre
    * @param r le rayon du cercle
    * @param n la nourriture a placer
    */
  public void placerNourriture(int x, int y, int r, Nourriture n)
  {
    placerNourriture(x, y, r, QUANTITE_NOURRITURE, n);
  }
  
  /**
    * Affichage d'un <tt>Ver</tt>. <P>
    *
    * @param v le <tt>Ver</tt> a dessiner
    */
  public void afficheVer(Ver v)
  {
    if((GestionSimulation.ETAT_COURANT != GestionSimulation.ACCELERER) && (GestionSimulation.ETAT_COURANT != GestionSimulation.ARRETER))
    {
      int i;
      if(v == null)
      {
        System.err.println(Erreur.getString("erreur_pistegraphique_affichever"));
        return ;
      }
      Point p = null;
      if(og == null)
        initGraphique();
      for(i = 0;i < v.getTaille();i++)
      {
        p = (Point)v.getPosition().elementAt(i);
        piste.setCaseVer(p);
        affichePoint(p, v.getColor());
      }
    }
  }
  
  /**
    * Permet de placer de la nourriture sur la piste de jeu,
    * aleatoirement. <P>
    *
    * Place un type <tt>t</tt> de nourriture de facon circulaire
    * sur la <tt>Piste</tt>. <p>
    *
    * @param x la coordonnee en X du centre
    * @param y la coordonnee en Y du centre
    * @param r la largeur du carre
    * @param nb la quantite de nourriture a placer
    * @param n la nourriture a placer
    */
  public void placerNourriture(int x, int y, int r, int nb, Nourriture n)
  {
    
    // cas particulier
    if(r <= 1)
    {
      if(piste.caseEstTraversable(x, y))
      {
        piste.setCase(x, y, n.getType());
        affichePoint(x, y, n.getCouleur());
      }
    }
    else
    {
      
      // Dans le but de ne pas faire une boucle infinie
      double maxTentatives = (r * r) - 1;
      int i;
      for(i = 0;((i <= nb) && (maxTentatives > 0));i++)
      {
        boolean trouve = false;
        int X = 0;
        int Y = 0;
        while(!trouve)
        {
          X = (Generateur.tirerIntEntre(r) * Generateur.signe()) + (x * Generateur.signe());
          Y = (Generateur.tirerIntEntre(r) * Generateur.signe()) + (y * Generateur.signe());
          trouve = (piste.caseEstTraversable(X, Y));
        }
        maxTentatives--;
        piste.setCase(X, Y, n.getType());
        affichePoint(X, Y, n.getCouleur());
      }
    }
    this.repaint();
  }
  
  /**
    * Methode qui renvoie true si la case (x,y) est MUR. <p>
    *
    * <b>ATTENTION !</b>
    * <ul>
    *   Toutes les cases hors de la <tt>Piste</tt> sont
    *   conciderees comme des MUR.
    * </ul>
    *
    * @param x Coordonnee en x du caractere
    * @param y Coordonnee en y du caractere
    *
    * @return
    * <ul>
    *   <li>true si la case contient MUR
    *   <li>false si la case contient autre chose ou sort des limites
    * </ul>
    */
  public boolean caseEstMur(int x, int y)
  {
    return piste.caseEstMur(x, y);
  }
  public void mouseReleased(MouseEvent e)
  {
    e.consume();
  }
  
  /**
    * Methode qui renvoie true si la case est MUR. <p>
    *
    * @param p <tt>Point</tt> qui serait MUR
    *
    * @return
    * <ul>
    *   <li>true si la case contient MUR
    *   <li>false si la case contient autre chose
    * </ul>
    */
  public boolean caseEstMur(Point p)
  {
    return piste.caseEstMur(p.x, p.y);
  }
  public void mouseClicked(MouseEvent e)
  {
    e.consume();
  }
  
  /**
    * Methode qui renvoie true si la case (x,y) est de la nourriture. <p>
    *
    * @param x Coordonnee en x du caractere
    * @param y Coordonnee en y du caractere
    *
    * @return
    * <ul>
    *   <li>true si la case contient de la nourriture
    *   <li>false si la case contient autre chose
    * </ul>
    */
  public boolean caseEstNourriture(int x, int y)
  {
    return piste.caseEstNourriture(x, y);
  }
  public void mouseExited(MouseEvent e)
  {
    e.consume();
  }
  
  /**
    * Methode qui renvoie true si la case est de la nourriture. <p>
    *
    * @param p <tt>Point</tt> qui serait de la Nourriture
    *
    * @return
    * <ul>
    *   <li>true si la case contient de la nourriture
    *   <li>false si la case contient autre chose
    * </ul>
    */
  public boolean caseEstNourriture(Point p)
  {
    return piste.caseEstNourriture(p.x, p.y);
  }
  
  /**
    * Methode qui renvoie true si la case est traversable. <p>
    *
    * Une case traversable est soit de la <b>Nourriture</b>
    * soit <b>VIDE</b>. <p>
    *
    * @param x Coordonnee en x du point
    * @param x Coordonnee en y du point
    *
    * @return
    * <ul>
    *   <li>true si la case est traversable
    *   <li>false si la case contient quelque chose de non traversable
    * </ul>
    */
  public boolean caseEstTraversable(int x, int y)
  {
    return piste.caseEstTraversable(x, y);
  }
  
  /**
    * Methode qui renvoie true si la case (x,y) est VIDE. <p>
    *
    * @param x Coordonnee en x du caractere
    * @param y Coordonnee en y du caractere
    *
    * @return
    * <ul>
    *   <li>true si la case contient VIDE
    *   <li>false si la case contient autre chose
    * </ul>
    */
  public boolean caseEstVide(int x, int y)
  {
    return piste.caseEstVide(x, y);
  }
  
  /**
    * Initialisation d'un objet <tt>PisteGraphique</tt>. <P>
    */
  public final void initGraphique()
  {
    if(GestionSimulation.ETAT_COURANT != GestionSimulation.ACCELERER)
    {
      oi = createImage(taillePix.width, taillePix.height);
      
      // oi est != null UNIQUEMENT si la PisteGraphique est VISIBLE
      if(oi != null)
      {
        og = oi.getGraphics();
        og.setColor(COULEUR_VIDE);
        og.fillRect(0, 0, taillePix.width - 1, taillePix.height - 1);
      }
      repaint();
    }
  }
  
  /**
    * Initialise la <tt>Piste</tt> et le dessin. <p> 
    */
  public final void initAll()
  {
    initGraphique();
    initPiste();
  }
  
  /**
    * Methode qui retourne la taille de la piste en Y. <P>
    *
    * @return la taille de la piste en Y
    */
  public final int getTailleY()
  {
    return piste.getTailleY();
  }
  
  /**
    * Methode qui retourne la taille de la piste en X. <P>
    *
    * @return la taille de la piste en X
    */
  public final int getTailleX()
  {
    return piste.getTailleX();
  }
  
  /**
    * Methode qui retourne la dimension de la piste. <P>
    *
    * Celle ci n'est pas forcement carre. <p>
    *
    * @return la dimension de la piste
    */
  public final Dimension getTaille()
  {
    return new Dimension(piste.getTaille());
  }
  
  /**
    * Nettoyage de la <tt>PisteGraphique</tt>. <P>
    */
  public final void clean()
  {
    if(GestionSimulation.ETAT_COURANT != GestionSimulation.ACCELERER)
    {
      if(oi != null)
      {
        og.setColor(COULEUR_VIDE);
        og.fillRect(0, 0, taillePix.width - 1, taillePix.height - 1);
      }
      else
      {
        initGraphique();
        clean();
      }
    }
  }
  
  /**
    * Initialisation de la <tt>Piste</tt> contenu dans la <tt>PisteGraphique</tt>. <P>
    */
  public final void initPiste()
  {
    piste.init();
  }
  
  /**
    * Retourne la <tt>Piste</tt> associee. <p>
    *
    *	@return la <tt>Piste</tt> associee.
    */
  public final Piste getPiste()
  {
    return piste;
  }
  
  /**
    * Methode qui entoure la <tt>PisteGraphique</tt> de MUR. <p>
    */
  protected void entourrer()
  {
    int i;
    
    // Placement des murs 
    for(i = 0;i < piste.getTailleX();i++)
    {
      setCaseMur(i, 0);
      setCaseMur(i, piste.getTailleY() - 1);
    }
    for(i = 0;i < piste.getTailleY();i++)
    {
      setCaseMur(0, i);
      setCaseMur(piste.getTailleX() - 1, i);
    }
  }
  
  /**
    * Affichage d'un point seul. <P>
    *
    * @param p le point
    * @param c la couleur du point
    */
  protected void affichePoint(Point p, Color c)
  {
    affichePoint(p.x, p.y, c);
  }
  
  /**
    * Affichage d'un point seul. <P>
    *
    * @param p le point
    * @param c la couleur du point
    */
  protected void affichePoint(int x, int y, Color c)
  {
    if(GestionSimulation.ETAT_COURANT != GestionSimulation.ACCELERER)
    {
      if(verifier(x, y))
      {
        if(og != null)
        {
          
          // dessiner le point puis appeler paint
          og.setColor(c);
          og.fillRect(x * mult, y * mult, mult - 1, mult - 1);
        }
        else
        {
          initGraphique();
          affichePoint(x, y, c);
        }
      }
    }
  }
  
  /**
    * Methode qui verifie les coordonnees. <p>
    *
    * @param x la coordonnees en X a verifier
    * @param y la coordonnees en Y a verifier
    */
  protected boolean verifier(int x, int y)
  {
    if((x > piste.getTailleX()) || (x < 0))
      return false;
    if((y > piste.getTailleY()) || (y < 0))
      return false;
    return true;
  }
  
  /**
    * Methode qui verifie les coordonnees. <p>
    *
    * @param p le <tt>Point</tt> a verifier
    */
  protected boolean verifier(Point p)
  {
    return verifier(p.x, p.y);
  }
  
  /** Dans le but de ne pas lier le package Simulation au package Ihm, mais
    * qu'au contraire le package Ihm soit lie au package Simulation, les deux
    * variables qui suivent permettent a la PisteGraphique de connaitre
    * les choix fait via la MenuBarre de la fenetre principale. <p>
    * 
    * Alors, oui, certe, ce n'est pas des plus fin, mais ca fonctionne bien,
    * tout en laissant sa liberte au package Simulation.
    */
  public static int TAILLE_PINCEAU = -1;
  
  /** Dans le but de ne pas lier le package Simulation au package Ihm, mais
    * qu'au contraire le package Ihm soit lie au package Simulation, les deux
    * variables qui suivent permettent a la PisteGraphique de connaitre
    * les choix fait via la MenuBarre de la fenetre principale. <p>
    * 
    * Alors, oui, certe, ce n'est pas des plus fin, mais ca fonctionne bien,
    * tout en laissant sa liberte au package Simulation.
    */
  public static int QUANTITE_NOURRITURE = 100;
  
  /** Dans le but de ne pas lier le package Simulation au package Ihm, mais
    * qu'au contraire le package Ihm soit lie au package Simulation, les deux
    * variables qui suivent permettent a la PisteGraphique de connaitre
    * les choix fait via la MenuBarre de la fenetre principale. <p>
    * 
    * Alors, oui, certe, ce n'est pas des plus fin, mais ca fonctionne bien,
    * tout en laissant sa liberte au package Simulation.
    */
  public static char TYPE_NOURRITURE = '*';
  
  /** Couleur des murs (<font color="black">Noirs</font>). <p> */
  public final static Color COULEUR_MUR = Color.black;
  
  /** Couleur du vide (<font color="blue">Bleu</font>). <p> */
  public final static Color COULEUR_VIDE = Color.blue;
  
  /** La taille, en pixel, du dessin. <p> */
  protected Dimension taillePix = null;
  
  /** La <tt>Piste</tt>. <p> */
  protected Piste piste = null;
  
  /** Taille, en pixel, d'une case. <p> */
  protected int mult = 3;
  
  /** Le <tt>Graphics</tt>. <p> */
  private Graphics og = null;
  
  /** La TableNourriture associee. <p> */
  private TableNourriture tableN = null;
  
  /** L'image. <p> */
  private Image oi = null;
}
