package fr.lifl.Aquarium.Simulation;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.awt.Color;
import java.awt.Dimension;

/**
  * <i>Classe</i> permettant de definir un objet <tt>Parametres</tt>. <P>
  *
  * Derniere modification 25 Fevrier 2000 <br>
  *
  * @author
  * <ul>
  *   <li>Hiblot Sebastien (V1.0)
  *   <li><a href="mailto:renaud91@hotmail.com">Ferret Renaud</a> (V2.++)
  * </ul>
  *
  * @version 2.6
  */
public class Parametres implements Serializable,Cloneable
{
  
  /**
    * Surcharge de la methode toString(). <p>
    *
    * Transforme un objet <tt>Parametres</tt> en une chaine de caracteres. <br>
    * Exemple : <br>
    * [ Cas etudie :INP, Nb de vers tues (%) :50, Taux de Mutation (% par gene) :5, Taux de CrossingOver (% par gene) :50, Longueur de la tranche (en nb de genes) :0, Nombre de Mutation (% sur les naissances) :10, Periode d'evolution :100, Nb de vers :5, Taille du ver :5, Types de nourriture :[ ('P', 10.0, red), ('N', 20.0, green) ], Taille piste :(250, 125) ]
    *
    * @return une String representant un objet <tt>Parametres</tt>
    */
  public String toString()
  {
    StringBuffer resu = new StringBuffer();
    resu.append("\n[ ");
    resu.append(Langage.getString("cas_etudie"));
    resu.append(cas);
    resu.append(",\n ");
    resu.append(Langage.getString("nb_vers_tues"));
    resu.append(nbVerTue);
    resu.append(",\n ");
    resu.append(Langage.getString("taux_mutation"));
    resu.append(tauxMutation);
    resu.append(",\n ");
    resu.append(Langage.getString("taux_crossingOver"));
    resu.append(tauxCrossingOver);
    resu.append(",\n ");
    resu.append(Langage.getString("taille_tranche"));
    resu.append(tailleTrancheCrossingOver);
    resu.append(",\n ");
    resu.append(Langage.getString("nb_mutation"));
    resu.append(nbMutation);
    resu.append(",\n ");
    resu.append(Langage.getString("periode_evolution"));
    resu.append(periodeEvolution);
    resu.append(",\n ");
    resu.append(Langage.getString("nb_vers"));
    resu.append(nbVer);
    resu.append(",\n ");
    resu.append(Langage.getString("taille_ver"));
    resu.append(tailleVer);
    resu.append(",\n ");
    resu.append(Langage.getString("tb_nourriture"));
    resu.append(tableNourriture);
    resu.append(", ");
    resu.append(Langage.getString("taille_piste"));
    resu.append("(");
    resu.append(taillePiste.width);
    resu.append(", ");
    resu.append(taillePiste.height);
    resu.append("),\n ");
    resu.append(Langage.getString("deux_fils"));
    resu.append(":");
    resu.append(deuxFils);
    resu.append(",\n ");
    resu.append(Langage.getString("cross_et_mute"));
    resu.append(":");
    resu.append(crossEtMute);
    resu.append(",\n ");
    resu.append(Langage.getString("nb_halo_fixe"));
    resu.append(":");
    resu.append(nbHaloFixe);
    resu.append(",\n ");
    resu.append(Langage.getString("coeff_ajustement_halo"));
    resu.append(":");
    resu.append(coeffAjustementHalo);
    resu.append(",\n ");
    resu.append(Langage.getString("couleur_halo"));
    resu.append(":");
    resu.append(couleurTypeHalo);
    resu.append(",\n ");
    resu.append(Langage.getString("min_halo"));
    resu.append(":");
    resu.append(minHalo);
    resu.append(",\n ");
    resu.append(Langage.getString("max_halo"));
    resu.append(":");
    resu.append(maxHalo);
    resu.append(",\n ");
    resu.append(Langage.getString("compter_diagonale"));
    resu.append(":");
    resu.append(diagonal);
    resu.append("] ");
    return resu.toString();
  }
  
  /**
    * Methode qui sauve le contenu des parametres. <P>
    *
    * @param nomFichier le nom du fichier ou sauver les parametres
    *
    * @return
    * <ul>
    *   <li>true si tout c'est bien passe
    *   <li>false sinon
    * </ul>
    */
  public boolean sauver(String nomFichier)
  {
    int i = 0;
    if(nomFichier == null)
      return false;
    if(nomFichier.trim().equals(""))
      return false;
    try
    {
      FileOutputStream fichierOuvert = new FileOutputStream(nomFichier);
      ObjectOutputStream out = new ObjectOutputStream(fichierOuvert);
      out.writeObject(this);
      out.flush();
      out.close();
      fichierOuvert.close();
    }
    catch(Exception e)
    {
      System.err.println(Erreur.getString("erreur_parametres_sauver"));
      return false;
    }
    return true;
  }
  
  /**
    * Constructeur d'un objet Parametres. <p>
    *
    * Les parametres par defaut sont ceux de la methode 
    * <tt>setDefaut()</tt>. <p>
    */
  public Parametres() 
  {
    setDefaut();
  }
  
  /**
    * Methode qui retourne une copie de l'objet. <p>
    *
    * @return une copie de l'objet.
    */
  public Object clone()
  {
    Parametres resu = new Parametres();
    resu.cas = new String(cas);
    resu.nbVerTue = nbVerTue;
    resu.tauxMutation = tauxMutation;
    resu.tauxCrossingOver = tauxCrossingOver;
    resu.nbMutation = nbMutation;
    resu.vitesse = vitesse;
    resu.periodeEvolution = periodeEvolution;
    resu.nbVer = nbVer;
    resu.tailleVer = tailleVer;
    resu.tailleTrancheCrossingOver = tailleTrancheCrossingOver;
    resu.tableNourriture = (TableNourriture)tableNourriture.clone();
    resu.taillePiste = new Dimension(taillePiste);
    resu.coutDeplacement = coutDeplacement;
    resu.periodeSevrage = periodeSevrage;
    resu.quantiteNourriture = quantiteNourriture;
    resu.deuxFils = deuxFils;
    resu.crossEtMute = crossEtMute;
    resu.nbHaloFixe = nbHaloFixe;
    resu.coeffAjustementHalo = coeffAjustementHalo;
    resu.minHalo = minHalo;
    resu.maxHalo = maxHalo;
    resu.couleurTypeHalo = couleurTypeHalo;
    resu.diagonal = diagonal;
    return resu;
  }
  
  /**
    * Methode qui modifie la taille du ver. <p>
    *
    * @param tailleVer la nouvelle taille du ver
    */
  public final void setTailleVer(int tailleVer)
  {
    if(tailleVer > 0)
      this.tailleVer = tailleVer;
  }
  
  /**
    * Methode qui modifie la taille de la tranche, utilisee lors des crossing overs. <p>
    *
    * Voir <tt>Simulation.Genotype.crossingOver(_)</tt> pour plus d'informations. <p>
    *
    * @param tTrancheCrossingOver la nouvelle taille de la tranche.
    */
  public final void setTailleTrancheCrossingOver(int tTrancheCrossingOver)
  {
    if(tTrancheCrossingOver > 0)
      this.tailleTrancheCrossingOver = tTrancheCrossingOver;
  }
  
  /**
    * Methode qui modifie la taille de la piste. <p>
    *
    * @param taillePiste la nouvelle taille de la piste
    */
  public final void setTaillePiste(Dimension taillePiste)
  {
    if((taillePiste.height > 0) && (taillePiste.width > 0))
      this.taillePiste = taillePiste;
  }
  
  /**
    * Retourne la periode (en nb de generations) entre deux sevrages. <p>
    * 
    * @return la periode (en nb de generations) entre deux sevrages. <p>
    */
  public final int getPeriodeSevrage()
  {
    return periodeSevrage;
  }
  
  /**
    * Retourne la taille de la tranche a prendre lors d'un crossing over. <p>
    *
    * Voir <tt>Simulation.Genotype.crossingOver(_)</tt> pour plus d'informations. <p>
    *
    * @return la taille de la tranche a prendre lors d'un crossing over.
    *
    * @see fr.lifl.Aquarium.Simulation.Genotype
    */
  public final int getTailleTrancheCrossingOver()
  {
    return tailleTrancheCrossingOver;
  }
  
  /**
    * Indique si les vers de halo fixe ont une couleur differente des vers de halo
    * proportionnel. <p>
    * 
    * @return
    *	<ul>
    *		<li>true, le type de halo influe sur la couleur du ver
    *		<li>false, le type de halo n'est pas lie a la couleur du ver
    *	</ul>
    */
  public final boolean getCouleurTypeHalo()
  {
    return couleurTypeHalo;
  }
  
  /**
    * Indique si les cases diagonales sont comptees deux ou une fois. <p>
    * 
    * @return
    *	<ul>
    *		<li>true, les cases diagonales sont comptees deux fois
    *		<li>false, les cases diagonales sont comptees une fois
    *	</ul>
    */
  public final boolean getDiagonal()
  {
    return diagonal;
  }
  
  /**
    * Methode qui retourne le pourcentage de naissances par mutation. <p>
    *
    * @return le pourcentage de naissances par mutation.
    */
  public final int getNbMutation()
  {
    return nbMutation;
  }
  
  /**
    * Methode qui modifie la taille de la piste. <p>
    *
    * @param taillePisteX la nouvelle taille de la piste en X
    * @param taillePisteY la nouvelle taille de la piste en Y
    */
  public final void setTaillePiste(int taillePisteX, int taillePisteY)
  {
    setTaillePiste(new Dimension(taillePisteY, taillePisteX));
  }
  
  /**
    * Retourne la vitesse actuelle de la simulation. <p>
    * 
    * @return la vitesse actuelle de la simulation
    */
  public final int getVitesse()
  {
    return vitesse;
  }
  
  /**
    * Methode qui modifie la table de nourritures. <p>
    *
    * Le cas etudie est remis a jour. <p>
    *
    * @param table la nouvelle table de nourritures
    */
  public final void setTableNourriture(TableNourriture table)
  {
    tableNourriture = table;
    if(table != null)
      cas = "I" + table.listerType();
    else
      cas = "";
  }
  
  /**
    * Retourne la valeur de crossEtMute. <p>
    *
    * @return
    * <ul>
    * 	<li>true si on applique  la fois la mutation et le crossing over sur les nouveaux vers
    * 	<li>false si l'on applique la mutation sur certains vers et le crossing sur d'autres.
    *	</ul>
    */
  public final boolean getCrossEtMute()
  {
    return crossEtMute;
  }
  
  /**
    * Modifie la quantite de nourritures qui sera donnee aux vers. <p>
    * 
    * @param quantite la nouvelle quantite de nourritures qui sera donnee aux vers.
    */
  public final void setQuantiteNourriture(int quantite)
  {
    quantiteNourriture = quantite;
  }
  
  /**
    * Retourne la valeur minimale que peut prendre un halo. <p>
    * 
    * @return la valeur minimale que peut prendre un halo.
    */
  public final int getMinHalo()
  {
    return minHalo;
  }
  
  /**
    * Modifie la periode entre deux sevrages. <p>
    * 
    * @param periode la nouvelle periode entre deux sevrages.
    */
  public final void setPeriodeSevrage(int periode)
  {
    if(periode > 0)
      periodeSevrage = periode;
  }
  
  /**
    * Permet de lier le type du halo avec une couleur de ver. <p>
    * 
    * @param b
    *	<ul>
    *		<li>true, le type de halo influe sur la couleur du ver
    *		<li>false, le type de halo n'est pas lie a la couleur du ver
    *	</ul>
    */
  public final void setCouleurTypeHalo(boolean b)
  {
    couleurTypeHalo = b;
    Genotype.COULEUR_HALO = b;
  }
  
  /**
    * Methode qui modifie la periode d'evolution. <p>
    *
    * Si p = 0, alors, il n'y aura plus d'evolution, la periode
    * sera infinie. <p>
    *
    * @param p la nouvelle periode d'evolution
    */
  public final void setPeriodeEvolution(int p)
  {
    if(p >= 0)
      this.periodeEvolution = p;
  }
  
  /**
    * Methode qui retourne la table de nourritures. <p>
    *
    * @return la table de nourritures.
    */
  public final TableNourriture getTableNourriture()
  {
    return tableNourriture;
  }
  
  /**
    * Methode qui modifie le nombre de vers tues par generation. <p>
    *
    * @param nbVerTue le nouveau nombre de vers tues par generation
    */
  public final void setNbVerTue(int nbVerTue)
  {
    if((nbVerTue > 0) || (nbVerTue <= 100))
      this.nbVerTue = nbVerTue;
  }
  
  /**
    * Retourne le taux de crossing over. <p>
    *
    * Pour plus d'informations, lire la methode <tt>crossingOver</tt> de <tt>Simulation.Genotype</tt>
    * 
    * @return e taux de crossing over (en %).
    *
    * @see fr.lifl.Aquarium.Simulation.Genotype
    */
  public final int getTauxCrossingOver()
  {
    return tauxCrossingOver;
  }
  
  /**
    * Methode qui modifie le nombre de vers initial. <p>
    *
    * @param nbVer le nombre de vers initial.
    */
  public final void setNbVer(int nbVer)
  {
    if(nbVer < 0)
      nbVer = 1;
    this.nbVer = nbVer;
  }
  
  /**
    * Methode qui modifie le cas initial. <p>
    *
    * @param cas le nouveau cas initial.
    */
  public final void setCas(String cas)
  {
    if(cas != null)
      this.cas = new String(cas);
  }
  
  /**
    * Methode qui modifie le pourcentage de naissances par mutation. <p>
    *
    * @param nbMutation le nouveau pourcentage de naissances par mutation.
    */
  public final void setNbMutation(int nbMutation)
  {
    if((nbMutation >= 0) || (nbMutation <= 100))
      this.nbMutation = nbMutation;
  }
  
  /**
    * Retourne la valeur d'un deplacement. <p>
    * 
    * @return la valeur d'un deplacement.
    */
  public final float getCoutDeplacement()
  {
    return coutDeplacement;
  }
  
  /**
    * Methode qui modifie le nombre de halos Fixes. <p>
    *
    * @param le nouveau nombre de halos Fixes, ou -1 si les types de halo
    * doivent etre generes aleatoirement.
    */
  public final void setNbHaloFixe(int nbHaloF)
  {
    if(nbHaloF <= 100)
      nbHaloFixe = nbHaloF;
  }
  
  /**
    * Methode qui retourne deuxFils. <p>
    * 
    * @return
    *	<ul>
    *		<li>true le crossing over donne deux fils
    *		<li>false le crossing over donne un fils
    *	</ul>
    */
  public final boolean getDeuxFils()
  {
    return deuxFils;
  }
  
  /**
    * Modifie la valeur minimale que peut prendre un halo. <p>
    * 
    * @param newMinHalo la nouvelle valeur minimale que peut prendre un halo.
    */
  public final void setMinHalo(int newMinHalo)
  {
    if((minHalo > 0) && (minHalo <= maxHalo))
    {
      minHalo = newMinHalo;
      Genotype.MIN_TAILLE_HALO = minHalo;
    }
  }
  
  /**
    * Retourne la valeur maximale que peut prendre un halo. <p>
    * 
    * @return la valeur maximale que peut prendre un halo.
    */
  public final int getMaxHalo()
  {
    return maxHalo;
  }
  
  /**
    * Modifie la valeur maximale que peut prendre un halo. <p>
    * 
    * @param newMaxHalo la nouvelle valeur maximale que peut prendre un halo.
    */
  public final void setMaxHalo(int newMaxHalo)
  {
    if((newMaxHalo > 0) && (maxHalo >= minHalo))
    {
      maxHalo = newMaxHalo;
      Genotype.MAX_TAILLE_HALO = maxHalo;
    }
  }
  
  /**
    * Methode qui retourne le nombre de halos Fixes. <p>
    *
    * @return le nombre de halos Fixes, ou -1 si les types de halo
    * doivent etre generes aleatoirement.
    */
  public final int getNbHaloFixe()
  {
    return nbHaloFixe;
  }
  
  /**
    * Fait en sorte que les cases diagonales soient comptees deux ou une fois. <p>
    * 
    * @param b
    *	<ul>
    *		<li>true, les cases diagonales seront comptees deux fois
    *		<li>false, les cases diagonales seront comptees une fois
    *	</ul>
    */
  public final void setDiagonal(boolean b)
  {
    diagonal = b;
    Ver.DIAGONALE = b;
  }
  
  /**
    * Methode qui retourne le nombre de vers initials. <p>
    *
    * @return le nombre de vers initial.
    */
  public final int getNbVer()
  {
    return nbVer;
  }
  
  /**
    * Methode qui modifie la valeur de deuxFils. <p>
    * 
    * @param b
    *	<ul>
    *		<li>true le crossing over donnera deux fils
    *		<li>false le crossing over donnera un fils
    *	</ul>
    */
  public final void setDeuxFils(boolean b)
  {
    deuxFils = b;
  }
  
  /**
    * Methode qui retourne la periode d'evolution. <p>
    *
    * @return la periode d'evolution.
    */
  public final int getPeriodeEvolution()
  {
    return periodeEvolution;
  }
  
  /**
    * Methode qui remet les parametres par defaut. <p>
    *
    * Parametre par defaut :
    * <ul>
    *   <li>vitesse=0
    *   <li>cas="INP"
    *   <li>croosEtMute = true
    *   <li>deuxFils = false
    *   <li>diagonal = true
    *   <li>couleurTypeHalo = true
    *   <li>minHalo = 4
    *   <li>maxHalo = 25
    *   <li>nbHaloFixe = -1, % de halo fixe, -1 => hazard
    *   <li>nbVerTue=50 %
    *   <li>tauxMutation=5 % par gene
    *   <li>nbMutation=5 % sur la population a naitre
    *   <li>tauxCrossingOver=50 % par gene
    *   <li>tailleTrancheCrossingOver=1 gene
    *   <li>coutDeplacement=0.5
    *   <li>coeffAjustementHalo = 0.05
    *   <li>periodeSevrage=5 en nb de generation
    *   <li>quantiteNourriture=200
    *   <li>periodeEvolution=100, 0=Infinie
    *   <li>nbVer=25
    *   <li>tailleVer=5
    *   <li>taillePiste = (300,150)
    *   <li>tableNourriture contient :
    *       <ul>
    *         <li>'N' de couleur <font color="green">verte</font> et de valeur 50
    *         <li>'P' de couleur <font color="red">rouge</font> et de valeur 10
    *       </ul>
    * </ul>
    */
  public final void setDefaut()
  {
    vitesse = 0;
    cas = "INP";
    crossEtMute = true;
    deuxFils = false;
    nbVerTue = 50;
    tauxMutation = 5;
    nbMutation = 5;
    tauxCrossingOver = 50;
    tailleTrancheCrossingOver = 1;
    periodeEvolution = 100;
    periodeSevrage = 5;
    quantiteNourriture = 200;
    coutDeplacement = (float)0.5;
    coeffAjustementHalo = (float)0.05;
    nbVer = 25;
    tailleVer = 5;
    taillePiste = new Dimension(300, 150);
    tableNourriture = new TableNourriture();
    Nourriture n = new Nourriture('N', 50);
    Nourriture p = new Nourriture('P', 10);
    n.setCouleur(Color.green);
    p.setCouleur(Color.red);
    tableNourriture.putNourriture(n);
    tableNourriture.putNourriture(p);
    nbHaloFixe = -1;
    setDiagonal(true);
    setCouleurTypeHalo(true);
    setMinHalo(4);
    setMaxHalo(25);
  }
  
  /**
    * Retourne la quantite de nourritures qui sera donnee aux vers. <p>
    * 
    * @return la quantite de nourritures qui sera donnee aux vers.
    */
  public final int getQuantiteNourriture()
  {
    return quantiteNourriture;
  }
  
  /**
    * Modifie la valeur de crossEtMute. <p>
    * 
    * @param b
    * <ul>
    * 	<li>true si on applique  la fois la mutation et le crossing over sur les nouveaux vers
    * 	<li>false si l'on applique la mutation sur certains vers et le crossing sur d'autres.
    *	</ul>
    */
  public final void setCrossEtMute(boolean b)
  {
    crossEtMute = b;
  }
  
  /**
    * Methode qui retourne la taille de la piste. <p>
    *
    * @return la taille de la piste
    */
  public final Dimension getTaillePiste()
  {
    return taillePiste;
  }
  
  /**
    * Methode qui modifie la probabilite qu'un gene soit mute. <p>
    *
    * @param tMutation la nouvelle probabilite qu'un gene soit mute
    */
  public final void setTauxMutation(int tMutation)
  {
    if((tMutation >= 0) || (tMutation <= 100))
      this.tauxMutation = tMutation;
  }
  
  /**
    * Methode qui retourne la taille du ver. <p>
    *
    * @return la taille du ver
    */
  public final int getTailleVer()
  {
    return tailleVer;
  }
  
  /**
    * Modifie la valeur d'un deplacement. <p>
    * 
    * @param cout le nouveau cout d'un deplacement
    */
  public final void setCoutDeplacement(float cout)
  {
    coutDeplacement = cout;
  }
  
  /**
    * Methode qui retourne la probabilite qu'un gene soit mute. <p>
    *
    * @return la probabilite qu'un gene soit mute
    */
  public final int getTauxMutation()
  {
    return tauxMutation;
  }
  
  /**
    * Methode qui modifie le coefficient d'ajustement du halo. <p>
    *
    * @param le nouveau coefficient d'ajustement du halo.
    */
  public final void setCoeffAjustementHalo(float coeff)
  {
    coeffAjustementHalo = coeff;
  }
  
  /**
    * Modifie la vitesse d'execution de la simulation. <p>
    * 
    * @param newVitesse la nouvelle vitesse de la simultaion. <p>
    */
  public final void setVitesse(int newVitesse)
  {
    if(newVitesse >= 0)
      vitesse = newVitesse;
  }
  
  /**
    * Methode qui retourne le cas initial. <p>
    *
    * @return le cas initial.
    */
  public final String getCas()
  {
    return cas;
  }
  
  /**
    * Modifie le taux de crossingOver (en %). <p>
    * 
    * @param tCrossingOver le nouveau taux de crossing over (%).
    */
  public final void setTauxCrossingOver(int tCrossingOver)
  {
    if((tCrossingOver >= 0) || (tCrossingOver <= 100))
      this.tauxCrossingOver = tCrossingOver;
  }
  
  /**
    * Methode qui retourne le coefficient d'ajustement du halo. <p>
    *
    * @return le coefficient d'ajustement du halo.
    */
  public final float getCoeffAjustementHalo()
  {
    return coeffAjustementHalo;
  }
  
  /**
    * Methode qui retourne le nombre de vers tues par generation. <p>
    *
    * @return le nombre de vers tues par generation
    */
  public final int getNbVerTue()
  {
    return nbVerTue;
  }
  
  /**
    * Charge les parametres. <P>
    *
    * @param nomFichier le nom du fichier d'ou charger les parametres
    *
    * @return
    * <ul>
    *   <li>le <tt>Parametres</tt> contenu dans le fichier
    *   <li>ou l'objet <tt>Parametres</tt> par defaut si nomFichier=null ou ="" ou ="Default"
    * </ul>
    */
  public static Parametres charger(String nomFichier)
  {
    Parametres p = null;
    if(nomFichier == null)
      return (new Parametres());
    if(nomFichier.trim().equals(""))
      return (new Parametres());
    if(nomFichier.equals("Default"))
      return (new Parametres());
    try
    {
      FileInputStream fichierOuvert = new FileInputStream(nomFichier);
      ObjectInputStream in = new ObjectInputStream(fichierOuvert);
      p = (Parametres)in.readObject();
      in.close();
      fichierOuvert.close();
      
      // Car il y a effet de bord
      p.setDiagonal(p.diagonal);
      p.setCouleurTypeHalo(p.couleurTypeHalo);
      p.setMinHalo(p.minHalo);
      p.setMaxHalo(p.minHalo);
    }
    catch(Exception e)
    {
      System.err.println(Erreur.getString("erreur_parametres_charger"));
      return (new Parametres());
    }
    return p;
  }
  
  /**
    * Indique le type d'algo utilise. <p>
    *
    * <ul>
    * 	<li>true si on applique  la fois la mutation et le crossing over sur les nouveaux vers
    * 	<li>false si l'on applique la mutation sur certains vers et le crossing sur d'autres.
    *	</ul>
    */
  protected boolean crossEtMute = true;
  
  /** Indique si le crossing over engendre deux (true) ou un (false) fils. <p> */
  protected boolean deuxFils = false;
  
  /** Periode d'evolution, en nombre de deplacement. <p> */
  protected int periodeEvolution = 0;
  
  /** Pourcentage de naissances par mutation. <p> */
  protected int nbMutation = 0;
  
  /** Decide du cas a etudier. <p> */
  protected String cas = null;
  
  /** Taille de la tranche pour les crossing over. <p> */
  protected int tailleTrancheCrossingOver = 0;
  
  /** Taille d'un ver. <p> */
  protected int tailleVer = 0;
  
  /** Indique si la couleur des vers depend de leur type de halo. <p> */
  protected boolean couleurTypeHalo = false;
  
  /** Probabilite qu'un gene soit mute. <p> */
  protected int tauxMutation = 0;
  
  /** Indique si les cases diagonales doivent etre compte deux fois ou une fois. <p> */
  protected boolean diagonal = true;
  
  /** Pourcentage de vers avec un halo Fixe. <p> 
    * 
    * Sachant que le nombre de halos proportionnels
    * sera 100 - nbHaloFixe. <p>
    */
  protected int nbHaloFixe = -1;
  
  /** Indique la valeur maximum d'un halo. <p> */
  protected int maxHalo = 25;
  
  /** Coefficient d'ajustement du halo. <p> */
  protected float coeffAjustementHalo = 1;
  
  /** Indique la valeur minimum d'un halo. <p> */
  protected int minHalo = 1;
  
  /** Nombre de vers au depart. <P> */
  protected int nbVer = 0;
  
  /** Indique la quantite de nourriture qui sera donnee au ver. <p> */
  protected int quantiteNourriture = 0;
  
  /** Vitesse d'execution (grand chiffre = tres lent). <p> */
  protected int vitesse = 0;
  
  /** Periode entre deux placement de nourriture. <p> */
  protected int periodeSevrage = 0;
  
  /** Probabilite qu'une tranche de genes vienne du pere ou de la mere. <p> */
  protected int tauxCrossingOver = 0;
  
  /** Cout d'un deplacement. <p> */
  protected float coutDeplacement = 0;
  
  /** Pourcentage de vers tues, pourcentage. <p> */
  protected int nbVerTue = 0;
  
  /** La taille de la piste. </P> */
  protected Dimension taillePiste = null;
  
  /** Table de nourritures. <p> */
  protected TableNourriture tableNourriture = null;
}
