package fr.lifl.Aquarium.Simulation;
import java.util.Vector;
import java.util.Enumeration;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.awt.Point;

/**
  * <i>Classe</i> permettant de definir une liste de <tt>Ver</tt>. <P>
  *
  * Cette objet permet une gestion plus transparente de la liste des <tt>Ver</tt>
  * utilisee dans l'<tt>Aquarium</tt>. <p>
  *
  * On peut la trier par exemple. <p>
  *
  * Derniere modification 27 fevrier 2000. <p>
  *
  * @author
  * <ul>
  *   <li>Hiblot Sebastien (V1.0)
  *   <li><a href="mailto:muriel_vidonne@yahoo.fr">Vidonne Muriel</a> (V2.3)
  *   <li><a href="mailto:renaud91@hotmail.com">Ferret Renaud</a> (V2.4)
  * </ul>
  *
  * @version 2.4
  *
  * @see fr.lifl.Aquarium.Simulation.Aquarium
  * @see fr.lifl.Aquarium.Simulation.Ver
  */
public class ListeVer implements Serializable
{
  
  /**
    * Constructeur d'un objet <tt>ListeVer</tt>. <p>
    */
  public ListeVer() 
  {
    listeVer = new Vector();
  }
  
  /**
    * Constructeur d'un objet <tt>ListeVer</tt>. <p>
    *
    * @param nbVer la capacite de cette liste
    */
  public ListeVer(int nbVer) 
  {
    if(nbVer <= 0)
      listeVer = new Vector();
    else
      listeVer = new Vector(nbVer);
  }
  
  /**
    */
  public void quickSort()
  {
    quickSort(0, listeVer.size() - 1);
  }
  
  /**
    * Retourne la plus grande taille du halo de type <tt>type</tt>. <p>
    *
    * @param type le type du halo
    *
    * @return la plus grande taille du halo de type <tt>type</tt>. <p>
    */
  public int getMaxHalo(char type)
  {
    int i;
    int nb = listeVer.size();
    int resu = Integer.MIN_VALUE;
    if(nb == 0)
      return 0;
    Ver v = null;
    for(i = 0;i < nb;i++)
    {
      v = getVer(i);
      if(v.getGenotype().getTypeHalo() == type)
        if(v.getGenotype().getTailleHalo() > resu)
          resu = v.getGenotype().getTailleHalo();
    }
    if(resu == Integer.MIN_VALUE)
      return 0;
    else
      return resu;
  }
  
  /**
    * Transforme un objet <tt>ListeVer</tt> en une <tt>String</tt>. <p>
    *
    * @return une <tt>String</tt> representant un <tt>ListeVer</tt>
    */
  public String toString()
  {
    StringBuffer resu = new StringBuffer();
    resu.append("[ NbVer = ");
    resu.append(listeVer.size());
    resu.append("\n, ");
    resu.append(listeVer.toString());
    resu.append("]");
    return resu.toString();
  }
  
  /**
    * Compte le nombre de halos de type <tt>type</tt>. <p>
    *
    * @param type le type de halo
    *
    * @return le nombre de halos de ce type
    */
  public int getNbHalo(char type)
  {
    int i;
    int nb = listeVer.size();
    int resu = 0;
    if(nb == 0)
      return 0;
    Ver v = null;
    for(i = 0;i < nb;i++)
    {
      v = getVer(i);
      if(v.getGenotype().getTypeHalo() == type)
        resu++;
    }
    return resu;
  }
  
  /**
    * Donne un pourcentage de similitude entre tous les adn des vers de la liste. <p>
    *
    * Typiquement, si j'ai 100 comme retour, et bien tous mes vers sont identiques (enfin, leur
    * adn du moins). <p>
    *
    * @return un pourcentage de similitude entre tous les adn des vers de la liste
    */
  public double degreConvergenceAdn()
  {
    int i, j, k;
    int nb = listeVer.size();
    double tab[] = new double[nb];
    double resu, nbGene, pas;
    Ver v1, v2;
    resu = 0;
    if(nb <= 1)
      return 100;
    for(i = 0;i < nb;i++)
    {
      tab[i] = 0;
      v1 = getVer(i);
      nbGene = v1.getGenotype().getAdn().getSize();
      for(j = 0;j < nb;j++)
      {
        if(j != i)
        {
          v2 = getVer(j);
          pas = 0;
          for(k = 0;k < nbGene;k++)
          {
            if(v1.getGenotype().getAdn().getValeurGene(k).equals(v2.getGenotype().getAdn().getValeurGene(k)))
              pas = pas + 1;
          }
          
          // Ici, je sais que le ver i a tab[i] gene en commun avec le ver j
          
          // je veux un pourcentage
          
          //				tab[i] = tab[i] + (pas / nbGene);
          tab[i] = tab[i] + pas;
        }
      }
      
      // Je veux savoir combien de gene le ver i a en commun avec tous les autres :
      tab[i] = tab[i] / ((nb - 1) * nbGene);
    }
    
    // Maintenant je vaus un indice global :
    for(i = 0;i < nb;i++)
      resu = resu + tab[i];
    resu = (resu / nb) * 100;
    
    // et voila
    return resu;
  }
  
  /**
    * Methode qui trie <tt>ListeVer</tt> en fonction de
    * la quantite d'energie de ceux-ci. <p>
    *
    * Les <tt>Ver</tt> qui possedent la vie la plus elevee sont
    * places en tete de la liste. <p>
    *
    * <b>ATTENTION !</b>
    * <ul>
    *   On refabrique ici une liste, c.a.d, les <tt>Ver</tt> de cette
    *   liste (<i>triee</i> <u>ne sont pas</u> ceux de la liste non triee.
    * </ul>
    */
  public void trier()
  {
    int i, j;
    Ver v1, v2;
    int taille = listeVer.size();
    if(taille == 0)
      return ;
    for(i = 0;i < taille;i++)
    {
      v1 = (Ver)(getVer(i)).clone();
      for(j = 0;j < taille;j++)
      {
        v2 = (Ver)(getVer(j)).clone();
        if(v1.getVie() > v2.getVie())
        {
          listeVer.setElementAt(v1, j);
          listeVer.setElementAt(v2, i);
          v1 = (Ver)(getVer(i)).clone();
        }
      }
    }
  }
  
  /**
    * Affiche toutes les informations sur tous les Vers de la liste. <p>
    *
    * @return une chaine representant ces informations.
    */
  public String afficherVers()
  {
    int i, j;
    StringBuffer s = new StringBuffer();
    synchronized(listeVer)
    {
      int taille = listeVer.size();
      if(taille == 0)
        return new String("");
      for(i = 0;i < taille;i++)
      {
        Ver v = getVer(i);
        s.append(v.toString());
      }
    }
    return s.toString();
  }
  
  /**
    * Retourne la taille moyenne du halo de type <tt>type</tt>. <p>
    *
    * @param type le type du halo
    *
    * @return la taille moyenne du halo de type <tt>type</tt>. <p>
    */
  public double getMoyenneHalo(char type)
  {
    int i, j = 0;
    int nb = listeVer.size();
    double resu = 0;
    Ver v = null;
    if(nb == 0)
      return 0;
    for(i = 0;i < nb;i++)
    {
      v = getVer(i);
      if(v.getGenotype().getTypeHalo() == type)
      {
        j++;
        resu = resu + v.getGenotype().getTailleHalo();
      }
    }
    if(j != 0)
      resu = resu / j;
    return resu;
  }
  
  /**
    * Methode qui sauve une <tt>ListeVer</tt>. <P>
    *
    * @param nomFichier le nom du fichier ou sauver la liste.
    *
    * @return
    * <ul>
    *   <li>true si tout c'est bien passe
    *   <li>false sinon
    * </ul>
    */
  public boolean sauver(String nomFichier)
  {
    int i = 0;
    if(nomFichier == null)
      return false;
    if(nomFichier.trim().equals(""))
      return false;
    try
    {
      FileOutputStream fichierOuvert = new FileOutputStream(nomFichier);
      ObjectOutputStream out = new ObjectOutputStream(fichierOuvert);
      out.writeObject(this);
      out.flush();
      out.close();
      fichierOuvert.close();
    }
    catch(Exception e)
    {
      System.err.println(Erreur.getString("erreur_listever_sauver"));
      return false;
    }
    return true;
  }
  
  /**
    * Affiche toutes les informations de convergence. <p>
    *
    * Cette methode ne fait pas appel a getMoyenneHalo, getNbHalo ..., ceci evite de parcourir
    * 8 fois la liste. Ici, toutes les informations sont collectees en un seul passage, gain de temps
    * et certitude de sortir les informations de la meme liste. <p>
    *
    * @return une chaine recapitulative.
    */
  public String informer()
  {
    
    // Cette methode ne fait pas appel a getMoyenneHalo, getNbHalo ..., ceci evite de parcourir
    
    // 8 fois la liste. Ici, toutes les informations sont collectees en un seul passage, gain de temps
    
    // et certitude de sortir les informations de la meme liste.
    int i, j, k;
    Ver v1, v2;
    double degCon = 0;
    double pas, nbGene;
    int maxHaloF = Integer.MIN_VALUE;
    int maxHaloP = Integer.MIN_VALUE;
    int minHaloF = Integer.MAX_VALUE;
    int minHaloP = Integer.MAX_VALUE;
    double moyHaloF = 0;
    double moyHaloP = 0;
    int nbHaloF = 0;
    int nbHaloP = 0;
    synchronized(listeVer)
    {
      int nb = listeVer.size();
      double tab[] = new double[nb];
      for(i = 0;i < nb;i++)
      {
        tab[i] = 0;
        v1 = getVer(i);
        nbGene = v1.getGenotype().getAdn().getSize();
        
        // Calcul des min, max et moy
        if(v1.getGenotype().getTypeHalo() == Genotype.HALO_FIXE)
        {
          nbHaloF++;
          if(v1.getGenotype().getTailleHalo() > maxHaloF)
            maxHaloF = v1.getGenotype().getTailleHalo();
          if(v1.getGenotype().getTailleHalo() < minHaloF)
            minHaloF = v1.getGenotype().getTailleHalo();
          moyHaloF = moyHaloF + v1.getGenotype().getTailleHalo();
        }
        if(v1.getGenotype().getTypeHalo() == Genotype.HALO_PROPORTIONNEL)
        {
          nbHaloP++;
          if(v1.getGenotype().getTailleHalo() > maxHaloP)
            maxHaloP = v1.getGenotype().getTailleHalo();
          if(v1.getGenotype().getTailleHalo() < minHaloP)
            minHaloP = v1.getGenotype().getTailleHalo();
          moyHaloP = moyHaloP + v1.getGenotype().getTailleHalo();
        }
        for(j = 0;j < nb;j++)
        {
          if(j != i)
          {
            v2 = getVer(j);
            pas = 0;
            for(k = 0;k < nbGene;k++)
            {
              if(v1.getGenotype().getAdn().getValeurGene(k).equals(v2.getGenotype().getAdn().getValeurGene(k)))
                pas = pas + 1;
            }
            
            // Ici, je sais que le ver i a tab[i] gene en commun avec le ver j
            
            // je veux un pourcentage
            
            //				tab[i] = tab[i] + (pas / nbGene);
            tab[i] = tab[i] + pas;
          }
        }
        
        // Je veux savoir combien de gene le ver i a en commun avec tous les autres :
        tab[i] = tab[i] / ((nb - 1) * nbGene);
      }
      
      // Maintenant je veux un indice global :
      for(i = 0;i < nb;i++)
        degCon = degCon + tab[i];
      degCon = (degCon / nb) * 100;
      
      // Cas particulier ou il n'y a qu'un ver
      if(nb <= 1)
        degCon = 100;
    }
    
    // Cas ou il n'y a pas de halo Proportionnel
    if(maxHaloP == Integer.MIN_VALUE)
      maxHaloP = 0;
    if(minHaloP == Integer.MAX_VALUE)
      minHaloP = 0;
    if(nbHaloP != 0)
      moyHaloP = moyHaloP / nbHaloP;
    
    // Cas ou il n'y a pas de halo fixe
    if(maxHaloF == Integer.MIN_VALUE)
      maxHaloF = 0;
    if(minHaloF == Integer.MAX_VALUE)
      minHaloF = 0;
    if(nbHaloF != 0)
      moyHaloF = moyHaloF / nbHaloF;
    
    // Affichage
    StringBuffer s = new StringBuffer();
    s.append("\nAdn = ");
    s.append(degCon);
    s.append(" (%)\nTypes Halo=(");
    s.append(Genotype.HALO_FIXE);
    s.append(" : ");
    s.append(nbHaloF);
    s.append(", ");
    s.append(Genotype.HALO_PROPORTIONNEL);
    s.append(" : ");
    s.append(nbHaloP);
    s.append(")\n\tFixe : Min = ");
    s.append(minHaloF);
    s.append(" Max = ");
    s.append(maxHaloF);
    s.append(" Moy = ");
    s.append(moyHaloF);
    s.append("\n\tProportionnel : Min = ");
    s.append(minHaloP);
    s.append(" Max = ");
    s.append(maxHaloP);
    s.append(" Moy = ");
    s.append(moyHaloP);
    return s.toString();
  }
  
  /**
    * Retourne la plus petite taille du halo de type <tt>type</tt>. <p>
    *
    * @param type le type du halo
    *
    * @return la plus petite taille du halo de type <tt>type</tt>. <p>
    */
  public int getMinHalo(char type)
  {
    int i;
    int nb = listeVer.size();
    int resu = Integer.MAX_VALUE;
    if(nb == 0)
      return 0;
    Ver v = null;
    for(i = 0;i < nb;i++)
    {
      v = getVer(i);
      if(v.getGenotype().getTypeHalo() == type)
        if(v.getGenotype().getTailleHalo() < resu)
          resu = v.getGenotype().getTailleHalo();
    }
    if(resu == Integer.MAX_VALUE)
      return 0;
    else
      return resu;
  }
  
  /**
    * Retourne le <tt>Ver</tt> contenu dans la 
    * <tt>ListeVer</tt> a la place <tt>index</tt>. <p>
    *
    * @param index l'emplacement du <tt>Ver</tt> dans la liste
    *
    * @return le <tt>Ver</tt> contenu dans la 
    * <tt>ListeVer</tt> a la place <tt>index</tt>.
    */
  public final Ver getVer(int index)
  {
    if((index >= 0) && (index < listeVer.size()))
      return (Ver)listeVer.elementAt(index);
    return null;
  }
  
  /**
    * Methode qui incremente le <tt>Compteur</tt> de <u>tous</u> 
    * les <tt>Ver</tt> de la <tt>ListeVer</tt>. <p>
    */
  public final void incAllGeneration()
  {
    int i;
    for(i = 0;i < listeVer.size();i++)
      getVer(i).incGeneration();
  }
  
  /**
    * Methode qui efface tous les <tt>Ver</tt> de la liste. <p>
    *
    * Un appel a <tt>tuerVer()</tt> est effectue sur chaque <tt>Ver</tt> 
    * avant de le retirer de la liste. <p>
    *
    * @see fr.lifl.Aquarium.Simulation.Ver
    */
  public final void tuerTous(PisteGraphique piste)
  {
    if(listeVer == null)
      return ;
    int i;
    Ver v;
    for(i = 0;i < listeVer.size();i++)
    {
      v = getVer(i);
      v.tuerVer(piste);
    }
    listeVer.removeAllElements();
  }
  
  /**
    * Methode qui reinitialise la vie de tous les vers a une valeur donnee. <p>
    */
  public final void initVie()
  {
    int i;
    for(i = 0;i < listeVer.size();i++)
      getVer(i).setVie(Ver.VIE_PAR_DEFAUT);
  }
  
  /**
    * Methode qui tue les derniers <tt>Ver</tt> de la <tt>ListeVer</tt>. <p>
    *
    * Un appel a <tt>tuerVer()</tt> est effectue sur chaque <tt>Ver</tt> 
    * avant de le retirer de la liste. <p>
    *
    * @param combien le nombre de <tt>Ver</tt> a tuer, en partant de la 
    * fin de la <tt>ListeVer</tt>
    *
    * @see fr.lifl.Aquarium.Simulation.Ver
    */
  public final void tuerVers(int combien, PisteGraphique piste)
  {
    
    // Attention, la taille change quand je fais un remove
    int i = listeVer.size() - 1;
    int nbTuer = 0;
    Ver v;
    
    // comme la taille change, il vaut mieux utiliser un while
    
    // qu'un for
    while((i >= 0) && (nbTuer != combien))
    {
      v = getVer(i);
      
      // Retirons les characteres VER, dans la piste
      v.tuerVer(piste);
      listeVer.removeElementAt(i);
      nbTuer++;
      i--;
    }
  }
  
  /**
    * Methode qui ajoute un <tt>Ver</tt> a la <tt>ListeVer</tt>. <p>
    *
    * @param v le <tt>Ver</tt> a ajouter a la <tt>ListeVer</tt>
    */
  public final void addVer(Ver v)
  {
    if(v != null)
      listeVer.add(v);
    else
      System.err.println(Erreur.getString("erreur_listever_addver"));
  }
  
  /**
    * Methode qui retire tous les <tt>Ver</tt> de la <tt>ListeVer</tt>. <p>
    *
    * Aucun appel a <tt>tuerVer()</tt> n'est effectue avant de le retirer de la liste. <p>
    *
    */
  public final void removeAll()
  {
    if(listeVer != null)
      listeVer.removeAllElements();
  }
  
  /**
    * Retourne le nombre de <tt>Ver</tt> contenus dans la 
    * <tt>ListeVer</tt>. <p>
    *
    * @return le nombre de <tt>Ver</tt> contenus dans la 
    * <tt>ListeVer</tt>.
    */
  public final int getSize()
  {
    return listeVer.size();
  }
  
  /**
    * Charge une <tt>ListeVer</tt>. <P>
    *
    * @param nomFichier le nom du fichier d'ou charger la liste
    *
    * @return
    * <ul>
    *   <li>la <tt>ListeVer</tt> contenue dans le fichier
    *   <li>null sinon
    * </ul>
    */
  public static ListeVer charger(String nomFichier)
  {
    if(nomFichier == null)
      return null;
    if(nomFichier.trim().equals(""))
      return null;
    if(nomFichier.equals("Default"))
      return null;
    try
    {
      FileInputStream fichierOuvert = new FileInputStream(nomFichier);
      ObjectInputStream in = new ObjectInputStream(fichierOuvert);
      ListeVer lv = (ListeVer)in.readObject();
      in.close();
      fichierOuvert.close();
      return lv;
    }
    catch(Exception e)
    {
      System.err.println(Erreur.getString("erreur_listever_charger"));
    }
    return null;
  }
  
  /** 
    * Fonction qui fait muter les vers. <P>
    * 
    * On fait muter nbMut vers choisis au hasard dans la liste des vers  muter. <p>
    *
    * @param parametres parametres de la simulation
    * @param pisteGraphique   
    * @param listeVers liste de vers  muter  
    * @param nbMut nombre de vers  muter
    * @param aAjouter
    *	<ul>
    *		<li>vrai : ajouter le ver  l'aquarium.
    *		<li>faux : ne pas l'ajouter 
    *	</ul>
    *
    * @return liste des vers muts
    */
  protected ListeVer mutation(Parametres parametres, PisteGraphique pisteGraphique, ListeVer versAMuter, int nbMut, boolean aAjouter)
  {
    int i, j;
    Ver v = null;
    Ver papa;
    ListeVer nouveauxNes = new ListeVer(nbMut);
    for(i = 0;i < nbMut;i++)
    {
      
      // Choix du ver parent
      j = Generateur.tirerIntEntre(versAMuter.getSize());
      papa = versAMuter.getVer(j);
      
      // Creation de l'enfant
      
      // dans le cas o un ver est  la fois crois et mut, on ne l'ajoute pas  l'aquarium
      
      // il y est dj, donc on ne fait que le muter
      if(aAjouter)
      { // on utilise le constructeur qui place le ver sur la piste			
        v = new Ver(pisteGraphique, parametres.getCas(), papa.getTaille());
        v.setGenotype((Genotype)papa.getGenotype().clone());
        
        // Mutation de l'enfant
        v.getGenotype().mute(parametres.getTauxMutation());
        
        // Ajout de l'enfant dans nouveauxNes
        nouveauxNes.addVer(v);
      }
      else
        
        // Mutation du ver :
        papa.getGenotype().mute(parametres.getTauxMutation());
    }
    return nouveauxNes;
  }
  
  /**
    * Fonction qui fait voluer les vers par crossing over et/ou par mutation. <P>
    * 
    * Permet d'utiliser diffrents algorithmes gntiques. <p>
    * 
    * @param parametres parametres de la simulation
    * @param pisteGraphique 
    * @param tues nombre de vers tus
    * @param deuxFils indique si le crossing over engendre 1 ou 2 fils
    * @param crossEtMute
    * <ul>
    *		<li>vrai : on applique  la fois la mutation et le crossing over
    *			sur les nouveaux vers
    *		<li>faux : on applique la mutation sur certains vers et le crossing
    *			sur d'autres
    *	</ul>
    */
  protected void evolution(Parametres parametres, PisteGraphique pisteGraphique, int tues, boolean deuxFils, boolean crossEtMute)
  {
    int i, j;
    int nbMut = 0;
    Ver v;
    ListeVer nouveauxNes = null;
    ListeVer nvxVers = null;
    
    // le nombre de mutation est un paramtre de la simulation
    nbMut = (tues * parametres.getNbMutation()) / 100;
    
    // les nouveaux vers subissent  la fois une mutation et un crossing over
    if(crossEtMute)
    {
      
      // si le crossing engendre 2 fils et qu'il y un nombre impair de 
      
      // vers  crer alors on en tue un de +
      
      //	System.out.println("Avt deuxFils");
      if(deuxFils && (tues % 2 != 0))
      {
        tuerVers(1, pisteGraphique);
        tues++;
      }
      
      // on commence par le crossing over : on croise 'tues' vers pour avoir un nombre 
      
      //de vers constant
      nouveauxNes = crossing(parametres, pisteGraphique, tues, deuxFils);
      
      // puis on fait nbMut mutations sur la liste des vers croiss :		
      mutation(parametres, pisteGraphique, nouveauxNes, nbMut, !crossEtMute);
      
      // on ajoute les nouveaux vers :
      for(i = 0;i < nouveauxNes.getSize();i++)
      {
        v = (Ver)(nouveauxNes.getVer(i)).clone();
        addVer(v);
      }
      nouveauxNes.removeAll();
    }
    
    //les nouveaux vers subissent soit une mutation, soit un crossing over
    else
    {
      
      // si le crossing engendre 2 fils et qu'il y un nombre impair de 
      
      // vers  croiser alors on en tue un de +
      if(deuxFils && ((tues - nbMut) % 2 != 0))
      {
        tuerVers(1, pisteGraphique);
        tues++;
      }
      
      // on commence par muter nbMut vers : ce sont des enfants des vers qu'on garde
      
      // (les parents ne sont pas muts, on les garde tels quels)
      nvxVers = mutation(parametres, pisteGraphique, this, nbMut, !crossEtMute);
      
      // puis on croise le nombre de vers adquat pour que le nombre de vers soit constant :
      
      // ce sont des enfants des vers qu'on garde (les parents ne sont pas croiss, 
      
      // on les garde tels quels)
      nouveauxNes = crossing(parametres, pisteGraphique, tues - nbMut, deuxFils);
      
      // on ajoute les vers mutes
      for(i = 0;i < nvxVers.getSize();i++)
      {
        v = (Ver)(nvxVers.getVer(i)).clone();
        addVer(v);
      }
      nvxVers.removeAll();
      
      // on ajoute les vers croises
      for(i = 0;i < nouveauxNes.getSize();i++)
      {
        v = (Ver)(nouveauxNes.getVer(i)).clone();
        addVer(v);
      }
      nouveauxNes.removeAll();
    }
  }
  
  /**
    * Fonction qui applique le crossing over sur les vers. <P>
    *   
    * On croise nbVers vers choisis au hasard dans la liste des vers. <p>
    * Il y a deux types de crossing over possibles :
    *	<ul>
    *		<li>soit on ne garde qu'un seul des 2 fils engendrs
    *		<li>soit on garde les 2 fils engendrs. 
    *	</ul>
    *
    * @param parametres parametres de la simulation
    * @param pisteGraphique 
    * @param nbVers le nombre de vers a croiser
    * @param deuxFils 
    *	<ul>
    *		<li>vrai, le crossing over engendre 2 fils
    *		<li>faux, il n'engendre qu'un seul fils
    *	</ul>
    *
    * @return liste des vers croises
    */
  protected ListeVer crossing(Parametres parametres, PisteGraphique pisteGraphique, int nbVers, boolean deuxFils)
  {
    int i, j, l;
    Ver papa, maman;
    Ver v, v2;
    int taille = listeVer.size();
    int nbCross = 0;
    if(deuxFils)
      nbCross = nbVers / 2;
    else
      nbCross = nbVers;
    ListeVer nouveauxNes = new ListeVer(nbCross);
    
    // On verifie qu'il y a au moins deux vers pour faire
    
    // du crossing over
    if(taille >= 2)
    {
      for(i = 0;i < nbCross;i++)
      {
        j = Generateur.tirerIntEntre(taille);
        l = j;
        while(j == l)
          l = Generateur.tirerIntEntre(taille);
        
        // Les parents
        papa = getVer(j);
        maman = getVer(l);
        
        // Creation du premier enfant :
        v = new Ver(pisteGraphique, parametres.getCas(), papa.getTaille());
        v.setGenotype((Genotype)(papa.getGenotype()).clone());
        
        // Creation du deuxime enfant :
        v2 = new Ver(pisteGraphique, parametres.getCas(), maman.getTaille());
        v2.setGenotype((Genotype)(maman.getGenotype()).clone());
        
        // Remarque : ce constructeur ajoute le ver cr sur la piste graphique si on ne le garde pas, il faudra l'enlever
        
        // Crossing over des enfants :
        Genotype gn2 = v.getGenotype().crossingOver(maman.getGenotype(), parametres.getTailleTrancheCrossingOver(), parametres.getTauxCrossingOver());
        v2.setGenotype(gn2);
        if(deuxFils)
        {
          
          // le crossing over engendre 2 fils
          
          // Ajout des 2 enfants dans nouveauxNes :
          nouveauxNes.addVer(v);
          nouveauxNes.addVer(v2);
        }
        else
        {
          
          // le crossing over engendre un seul fils
          
          //on choisit au hasard le fils issu du crossing over :
          if(Generateur.tirerIntEntre(2) == 1)
          {
            
            //on enlve de la piste le ver qu'on ne garde pas
            v2.tuerVer(pisteGraphique);
            
            // Ajout de l'enfant dans nouveauxNes :
            nouveauxNes.addVer(v);
          }
          else
          {
            
            //on enlve de la piste le ver qu'on ne garde pas
            v.tuerVer(pisteGraphique);
            
            // Ajout de l'enfant dans nouveauxNes :
            nouveauxNes.addVer(v2);
          }
        }
      }
    }
    return nouveauxNes;
  }
  
  /**
    * Trie rapide QuickSort d'une liste de ver par order decroissant de niveau d'energie. <p>
    *
    * Le ver le plus energetique est en tete de liste. <p>
    *
    * @param g <b>doit</b> etre a zero.
    * @param d <b>doit</b> etre a N-1, N = size().
    */
  private void quickSort(int g, int d)
  {
    int i = g, j = d;
    Ver u, v;
    int taille = listeVer.size();
    boolean b = true;
    if(d > g)
    {
      v = (Ver)(getVer(d)).clone();
      while(b)
      {
        for(i = g;i < taille;i++)
        {
          if(getVer(i).getVie() <= v.getVie())
            break;
        }
        for(j = d;j >= 0;j--)
        {
          if(getVer(j).getVie() > v.getVie())
            break;
        }
        if(i >= j)
          break;
        u = getVer(i);
        listeVer.setElementAt(getVer(j), i);
        listeVer.setElementAt(u, j);
      }
      u = getVer(i);
      listeVer.setElementAt(getVer(d), i);
      listeVer.setElementAt(u, d);
      quickSort(g, i - 1);
      quickSort(i + 1, d);
    }
  }
  
  /** Les <tt>Ver</tt> seront ranges dans ce <tt>Vector</tt>. <p> */
  protected Vector listeVer = null;
}
