package fr.lifl.Aquarium.Simulation;
import java.util.Hashtable;
import java.util.Enumeration;
import java.io.Serializable;
import java.awt.Color;

/**
  * <i>Classe</i> permettant de definir l'objet <tt>Genotype</tt>. <P>
  *
  * Celui-ci caracterise :
  * <ul>
  *   <li>le comportement d'un <tt>Ver</tt> via son <tt>Adn</tt>
  *   <li>le type de halo d'un <tt>Ver</tt>
  *   <li>la taille du halo d'un <tt>Ver</tt>
  *   <li>la taille d'un <tt>Ver</tt>
  *   <li>la couleur d'un <tt>Ver</tt>
  * </ul>
  *
  * Derniere modification 22 Janvier 2000. <p>
  *
  * @author
  * <ul>
  *   <li>Hiblot Sebastien (V1.0)
  *   <li><a href="mailto:renaud91@hotmail.com">Ferret Renaud</a>
  *   <li><a href="mailto:muriel_vidonne@yahoo.fr">Vidonne Muriel</a> (V2.2)
  * </ul>
  *
  * @version 2.2
  *
  * @see fr.lifl.Aquarium.Simulation.Adn
  * @see fr.lifl.Aquarium.Simulation.Ver
  */
public class Genotype implements Serializable,Cloneable
{
  
  /**
    * Transforme un objet <tt>Genotype</tt> en une <tt>String</tt>. <p>
    *
    * @return une <tt>String</tt> representant un <tt>Genotype</tt>
    */
  public String toString()
  {
    StringBuffer res = new StringBuffer();
    res.append("[ ");
    res.append(Langage.getString("taille_ver"));
    res.append(tailleVer);
    res.append(", ");
    res.append(Langage.getString("taille_halo"));
    res.append(tailleHalo);
    res.append(", ");
    res.append(Langage.getString("type_halo"));
    res.append(typeHalo);
    res.append(", Adn=");
    res.append(adn);
    
    //	res.append(", ");
    
    //	res.append(Langage.getString("couleur"));
    
    //	res.append(couleur);
    res.append("]");
    return res.toString();
  }
  
  /**
    * Methode qui retourne une copie de l'objet. <p>
    *
    * @return une copie de l'objet.
    */
  public Object clone()
  {
    Genotype resu = new Genotype();
    resu.adn = (Adn)adn.clone();
    resu.tailleHalo = tailleHalo;
    resu.typeHalo = typeHalo;
    resu.tailleVer = tailleVer;
    resu.couleur = couleur;
    return resu;
  }
  
  /**
    * Affiche l'ADN du Ver. <p>
    *
    * Pour tester le crossing over :  affiche les gnes selon l'ordre 
    * dans lequel on les parcourt. <p>
    *
    */
  public void afficheAdn()
  {
    adn.affiche();
  }
  
  /**
    * Methode qui fait un crossing over entre deux Genotypes. <p>
    *
    * <b>ATTENTION !</b> Il ne faut pas oublier que si l'on veut faire muter
    * la taille du <tt>Ver</tt>, celui-ci possede <u>DEJA</u> un <tt>Vector</tt>
    * de position. <p>
    *
    * @param gen le second genotype parent.
    *
    * @see fr.lifl.Aquarium.Simulation.Parametres
    * @see fr.lifl.Aquarium.Simulation.Adn
    */
  public void crossingOver(Genotype gen)
  {
    crossingOver(gen, 1, 50);
  }
  
  /**
    * Constructeur de la classe <tt>Genotype</tt>. <p>
    *
    * Les types de <tt>Nourriture</tt> sont tires de la <tt>String cas</tt>. <p>
    *
    * @param tailleVer la taille du <tt>Ver</tt>
    * @param cas le cas etudie ("PINM" par exemple)
    *
    * @see fr.lifl.Aquarium.Simulation.Adn
    */
  public Genotype(int tailleVer, String cas) 
  {
    this(cas);
    this.setTailleVer(tailleVer);
  }
  
  /**
    * Methode qui fait muter un certain pourcentage 
    * des genes de l'<tt>Adn</tt>. <p>
    *
    * <b>ATTENTION !</b> Il ne faut pas oublier qu si l'on veut faire muter
    * la taille du <tt>Ver</tt>, celui-ci possede <u>DEJA</u> un <tt>Vector</tt>
    * de position. <p>
    *
    * @param tauxMutation le taux de mutation
    *
    * @see fr.lifl.Aquarium.Simulation.Parametres
    * @see fr.lifl.Aquarium.Simulation.Adn
    */
  public void mute(int tauxMutation)
  {
    int i, j;
    
    // Mutation de la taille du halo
    i = Generateur.tirerIntEntre(100);
    if(i <= tauxMutation)
      tailleHalo = tirerTailleHalo();
    
    // Mutation du type du halo	
    i = Generateur.tirerIntEntre(100);
    if(i <= tauxMutation)
      typeHalo = tirerTypeHalo();
    colorier();
    
    // Mutation d'un gene
    for(j = 1;j <= adn.getSize();j++)
    {
      
      // Probabilite qu'un gene mute  
      i = Generateur.tirerIntEntre(100);
      if(i <= tauxMutation)
        adn.mute(j);
    }
  }
  
  /**
    * Constructeur de la classe <tt>Genotype</tt>. <p>
    *
    * Les types de <tt>Nourriture</tt> sont tires de la <tt>String cas</tt>. <p>
    *
    * @param tailleHalo la taille du halo
    * @param typeHalo le type du halo
    * @param cas le cas etudie ("PINM" par exemple)
    * @param tailleVer la taille du <tt>Ver</tt>
    *
    * @see fr.lifl.Aquarium.Simulation.Adn
    */
  public Genotype(int tailleHalo, char typeHalo, String cas, int tailleVer) 
  {
    this.tailleHalo = tailleHalo;
    this.typeHalo = typeHalo;
    adn = new Adn(cas);
    setTailleVer(tailleVer);
    colorier();
  }
  
  /**
    * Constructeur de la classe <tt>Genotype</tt>. <p>
    *
    * Les types de <tt>Nourriture</tt> sont tires de la <tt>String cas</tt>. <p>
    *
    * @param tailleVer la taille du <tt>Ver</tt>
    * @param typeHalo le type du halo
    * @param cas le cas etudie ("PINM" par exemple)
    *
    * @see fr.lifl.Aquarium.Simulation.Adn
    */
  public Genotype(int tailleVer, char typeHalo, String cas) 
  {
    this(tirerTailleHalo(), typeHalo, cas, tailleVer);
  }
  
  /**
    * Constructeur de la classe <tt>Genotype</tt>. <p>
    *
    * Les types de <tt>Nourriture</tt> sont tires de la <tt>String cas</tt>. <p>
    *
    * @param cas le cas etudie ("PINM" par exemple)
    */
  public Genotype(String cas) 
  {
    this(tirerTailleHalo(), tirerTypeHalo(), cas, 0);
  }
  
  /**
    * Methode qui fait un crossing over entre deux Genotypes par tranche. <p>
    *
    * Pour avoir un crossing over uniforme il suffit de mettre la taille de la 
    * tranche  1. <p>
    *
    * <b>ATTENTION !</b> Il ne faut pas oublier que si l'on veut faire muter
    * la taille du <tt>Ver</tt>, celui-ci possede <u>DEJA</u> un <tt>Vector</tt>
    * de position. <p>
    *
    * @param gen le second genotype parent.
    * @param tailleTranche taille des tranches  "croiser"
    * @param tauxCrossing probabilite (%) d'heriter de this, c.a.d que this equivalent a
    * a gen a tauxCrossing (%) pres.
    *
    * @return retourne le gnotype 2eme fils issu du crossing over
    * 
    * @see fr.lifl.Aquarium.Simulation.Parametres
    * @see fr.lifl.Aquarium.Simulation.Adn
    */
  public Genotype crossingOver(Genotype gen, int tailleTranche, int tauxCrossing)
  {
    int i, j, k, nb, pos = 1;
    Genotype gn = (Genotype)gen.clone();
    String cas;
    
    // Taille du halo :
    i = Generateur.tirerIntEntre(100);
    if(i < tauxCrossing)
    {
      gn.tailleHalo = tailleHalo;
      tailleHalo = gen.tailleHalo;
    }
    
    // Type de halo :
    i = Generateur.tirerIntEntre(100);
    if(i < tauxCrossing)
    {
      gn.typeHalo = typeHalo;
      typeHalo = gen.typeHalo;
    }
    colorier();
    gn.colorier();
    
    // Cas Moyen :
    nb = adn.getSize() / tailleTranche;
    for(j = 0;j < nb;j++)
    {
      i = Generateur.tirerIntEntre(100);
      if(i < tauxCrossing)
      {
        for(k = 1;k <= tailleTranche;k++)
        {
          pos = (j * tailleTranche) + k;
          cas = adn.getValeurGene(pos);
          adn.setValeurGene(pos, gen.adn.getValeurGene(pos));
          gn.adn.setValeurGene(pos, cas);
        }
      }
    }
    
    // derniers gnes pour le cas o la division n'tait pas exacte
    pos = nb * tailleTranche;
    if(pos < adn.getSize())
    {
      i = Generateur.tirerIntEntre(100);
      if(i < tauxCrossing)
      {
        for(k = pos + 1;k <= adn.getSize();k++)
        {
          cas = adn.getValeurGene(k);
          adn.setValeurGene(k, gen.adn.getValeurGene(k));
          gn.adn.setValeurGene(k, cas);
        }
      }
    }
    return gn;
  }
  
  /**
    * Methode qui retourne le cas etudie. <p>
    *
    * Le cas etudie est tire de l'<tt>Adn</tt> du <tt>Genotype</tt>. <p>
    *
    * @return le cas etudie. (exp "PINM")
    */
  public final String getCas()
  {
    return adn.getCas();
  }
  
  /**
    * Methode retournant l'<tt>Adn</tt> du <tt>Genotype</tt>. <P>
    *
    * @return l'<tt>Adn</tt> du <tt>Genotype</tt>
    */
  public final Adn getAdn()
  {
    return adn;
  }
  
  /**
    * Teste si le halo est fixe ou non. <p>
    * 
    * @return
    * <ul>
    *		<li>true si le halo est fixe
    *		<li>false sinon
    * </ul>
    */
  public final boolean haloEstFixe()
  {
    return (typeHalo == HALO_FIXE);
  }
  
  /**
    * Retourne la couleur du ver. <p>
    * 
    * @return la couleur du ver
    */
  public final Color getCouleur()
  {
    return couleur;
  }
  
  /**
    * Methode qui retourne le type de halo du <tt>Ver</tt>. <P>
    *
    * @return le type halo du <tt>Ver</tt>
    */
  public final char getTypeHalo()
  {
    return typeHalo;
  }
  
  /**
    * Methode qui change la taille du halo. <P>
    *
    * @param la nouvelle taille du halo
    */
  public final void setTailleHalo(int t)
  {
    if(t > 0)
      tailleHalo = t;
  }
  
  /**
    * Modifie la taille du <tt>Ver</tt>. <p>
    *
    * @param taille la nouvelle taille du <tt>Ver</tt>
    */
  public final void setTailleVer(int taille)
  {
    if(taille > 0)
      tailleVer = taille;
  }
  
  /**
    * Methode qui modifie le type de halo. <P>
    *
    * @param le nouveau type de halo
    */
  public final void setTypeHalo(char t)
  {
    typeHalo = t;
    colorier();
  }
  
  /**
    * Teste si le halo est proportionnel ou non. <p>
    * 
    * @return
    * <ul>
    *		<li>true si le halo est proportionnel
    *		<li>false sinon
    * </ul>
    */
  public final boolean haloEstProportionnel()
  {
    return (typeHalo == HALO_PROPORTIONNEL);
  }
  
  /**
    * Modifie la couleur du ver. <p>
    * 
    * @param newCouleur la nouvelle couleur du ver
    */
  public final void setCouleur(Color newCouleur)
  {
    couleur = newCouleur;
  }
  
  /**
    * Retourne la taille du <tt>Ver</tt>. <p>
    *
    * @return la taille du <tt>Ver</tt>
    */
  public final int getTailleVer()
  {
    return tailleVer;
  }
  
  /**
    * Methode qui retourne la taille du halo du <tt>Ver</tt>. <P>
    *
    * @return la taille halo du <tt>Ver</tt>
    */
  public final int getTailleHalo()
  {
    return tailleHalo;
  }
  
  /**
    * Constructeur sans parametre de <tt>Genotype</tt>. <p>
    *
    * Ce constructeur ne <b><u>doit rien faire</u></b>, il sert uniquement dans
    * la methode <tt>clone()</tt>. <p>
    */
  protected Genotype() 
  {
    ;
  }
  
  /** 
    * Change la couleur du ver en fonction du type de son halo. <p>
    * Creation date: (07/03/00 12:52:58)
    */
  protected final void colorier()
  {
    if(COULEUR_HALO)
    {
      if(haloEstFixe())
        setCouleur(COULEUR_HALO_FIXE);
      if(haloEstProportionnel())
        setCouleur(COULEUR_HALO_PROPORTIONNEL);
    }
  }
  
  /**
    * Tire au hasard le type de halo. <p>
    *
    * @return le nouveau type de halo.
    */
  protected static char tirerTypeHalo()
  {
    if(Generateur.tirerIntEntre(2) == 0)
      return HALO_FIXE;
    else
      return HALO_PROPORTIONNEL;
  }
  
  /**
    * Tire la capacite du halo. <p>
    *
    * La valeur est comprise entre <tt>MIN_TAILLE_HALO</tt> et <tt>MAX_TAILLE_HALO</tt>. <p>
    *
    * @return la capacite de vision du halo.
    */
  protected static int tirerTailleHalo()
  {
    return Generateur.tirerIntEntre(MIN_TAILLE_HALO, MAX_TAILLE_HALO);
  }
  
  /** Sert a fixer ou non la couleur des vers selon leur type de halo. <p> */
  public static boolean COULEUR_HALO = true;
  
  /** Indique la taille minimum du halo. <p> */
  public static int MIN_TAILLE_HALO = 4;
  
  /** Indique la taille maximum du halo. <p> */
  public static int MAX_TAILLE_HALO = 25;
  
  /** Indique que le Halo est proportionnel. <p> */
  public final static char HALO_PROPORTIONNEL = 'P';
  
  /** Couleur par defaut des <tt>Ver</tt> ayant un <font color="white">halo fixe</font>. <p> */
  public final static Color COULEUR_HALO_FIXE = Color.white;
  
  /** Couleur par defaut des <tt>Ver</tt> ayant un <font color="orange">halo proportionnel</font>. <p> */
  public final static Color COULEUR_HALO_PROPORTIONNEL = Color.orange;
  
  /** Indique que le Halo est fixe. <p> */
  public final static char HALO_FIXE = 'F';
  
  /** La couleur du ver. <p> */
  protected Color couleur = Color.white;
  
  /** La taille du <tt>Ver</tt>. <p> */
  protected int tailleVer = 0;
  
  /** La taille du halo. <p> */
  protected int tailleHalo;
  
  /** Type du halo. <p> */
  protected char typeHalo;
  
  /** L'<tt>Adn</tt> du <tt>Genotype</tt>. <p> */
  protected Adn adn = null;
}
