package fr.lifl.Aquarium.Simulation;
import java.util.Hashtable;
import java.util.Enumeration;
import java.io.Serializable;

/**
  *
  * Cette <i>classe</i> represente un <tt>Adn</tt> de <tt>Ver</tt>. <p>
  *
  * L'<tt>Adn</tt> permet de coder les deplacements du <tt>Ver</tt>. C'est 
  * grace a lui que le <tt>Ver</tt> sait (ou ne sait pas) ou il doit aller. <p>
  *
  * Un <tt>Adn</tt> se compose de deux choses :
  * <ul>
  *   <li>ce qu'il voit (la clef de la <tt>Hashtable</tt>
  *   <li>la ou il va
  * </ul>
  *
  * Exemple :<br>
  * Si un <tt>Ver</tt> voit "III", et que son deplacement lui dit "FGD", le 
  * <tt>Ver</tt> suivra le deplacement suivant : si il peut aller en <b>F</b>ace, 
  * il y va, sinon il ira a <b>G</b>auche, et si il ne peut ni aller en Face ni
  * aller a Gauche il ira a <b>D</b>roite. <p>
  *
  * Un objet <tt>Adn</tt> est contenu dans un objet <tt>Genotype</tt>. <p>
  *
  * Il est maintenant possible de generer un <tt>Adn</tt> parfait, on definit 
  * un <tt>Adn</tt> parfait :
  * <ul>
  *   <li>le <tt>Ver</tt> doit se diriger ver la <tt>Nourriture</tt> la
  *       plus energetique.
  *   <li>il doit favoriser le deplacement Face, en cas d'exequo.
  * </ul>
  *
  * Exemple d'un <tt>Adn</tt> parfait :<br>
  * Si on prend 2 types de nourritures : ('N',100) et ('P',0).<br>
  * Alors
  * <ul>
  *   <li>"III" => "FGD"
  *   <li>"NII" => "GFD"
  *   <li>"INI" => "GFD"
  *   <li>"NNI" => "FGD"
  *   <li>"PNP" => "FGD"
  *   <li>"PII" => "GFD"
  *   <li>...
  * </ul>
  *
  * Derniere modification 15 Mars 2000. <p>
  *
  * @author
  * <ul>
  *   <li>Hiblot Sebastien (V1.0)
  *   <li><a href="mailto:renaud91@hotmail.com">Ferret Renaud</a> (V2.++)
  * </ul>
  *
  * @version 2.3
  *
  * @see fr.lifl.Aquarium.Simulation.Genotype
  * @see fr.lifl.Aquarium.Simulation.Ver
  */
public class Adn implements Serializable,Cloneable
{
  
  /**
    * Transforme un objet <tt>Adn</tt> en une <tt>String</tt>. <p>
    *
    * @return une <tt>String</tt> representant un <tt>Adn</tt>
    */
  public String toString()
  {
    StringBuffer resu = new StringBuffer();
    resu.append("[ Nb = ");
    resu.append(adn.size());
    resu.append(", ");
    resu.append(adn.toString());
    return resu.toString();
  }
  
  /**
    * Modifie le gene. <p>
    *
    * @param gene le gene a modifier
    * @param nouvelleValeur la nouvelle valeur du gene
    */
  public void setValeurGene(String gene, String nouvelleValeur)
  {
    if(adn.remove(gene) == null)
    {
      System.err.println(Erreur.getString("erreur_adn_setvaleurgene"));
      return ;
    }
    adn.put(gene, nouvelleValeur);
  }
  
  /**
    * Modifie le gene. <p>
    *
    * @param qui le numero du gene a modifier
    * @param nouvelleValeur la nouvelle valeur du gene
    */
  public void setValeurGene(int qui, String nouvelleValeur)
  {
    setValeurGene(convert(qui), nouvelleValeur);
  }
  
  /**
    * Methoque qui fait muter un gene. <p>
    *
    * Pour faire muter un gene, on retire celui-ci au hasard. <p>
    *
    * @param le gene a muter
    */
  public void mute(String gene)
  {
    
    // On retire le gene
    if(adn.remove(gene) == null)
    {
      System.err.println(Erreur.getString("erreur_adn_mute"));
      return ;
    }
    
    // et on ajoute le nouveau
    adn.put(gene, dirRandom());
  }
  
  /**
    * Methoque qui fait muter un gene. <p>
    *
    * Pour faire muter un gene, on retire celui-ci au hasard. <p>
    *
    * @param le numero du gene a muter
    */
  public void mute(int qui)
  {
    mute(convert(qui));
  }
  
  /**
    * Methode qui retourne la valeur d'un gene. <p>
    *
    * @param qui le numero du gene
    *
    * @return son contenu (le deplacement associe, "FGD" par exemple)
    */
  public String getValeurGene(int qui)
  {
    return getValeurGene(convert(qui));
  }
  
  /**
    * Methode qui retourne une copie de l'objet. <p>
    *
    * @return une copie de l'objet.
    */
  public Object clone()
  {
    Adn resu = new Adn();
    resu.adn = (Hashtable)adn.clone();
    resu.cas = new String(cas);
    resu.estParfait = estParfait;
    return resu;
  }
  
  /**
    * Rend l'<tt>Adn</tt> parfait. <p>
    *
    * Ici, le but est de generer un <tt>Adn</tt> parfait,
    * (Voir plus haut pour la definition d'un <tt>Adn</tt> parfait). <p>
    *
    * @param la <tt>TableNourriture</tt> associee a la <i>perfection</i> de l<tt>Adn</tt>
    */
  public void rendreAdnParfait(TableNourriture tn)
  {
    int i;
    int nbCas = cas.length();
    for(i = 1;i <= Math.pow(nbCas, 3);i++)
    {
      String s = convert(i);
      setValeurGene(s, rendreGeneParfait(s, tn));
    }
  }
  
  /**
    * Constructeur d'un <tt>Adn</tt>. <p>
    *
    * Les lettres utilisees pour generer un <tt>Adn</tt> sont extraites
    * de la <tt>TableNourriture</tt>, le "I" (pour Inmengeable) est ajoute 
    * automatiquement. <p>
    *
    * @param tn la <tt>TableNourriture</tt> d'ou vont etre extraites les
    * lettres constituant les nourritures.
    */
  public Adn(TableNourriture tn) 
  {
    this("I" + tn.listerType());
  }
  
  /**
    * Methode qui retourne la valeur d'un gene. <p>
    *
    * @param gene le gene
    *
    * @return son contenu (le deplacement associe, "FGD" par exemple)
    */
  public String getValeurGene(String gene)
  {
    return (String)adn.get(gene);
  }
  
  /**
    * Constructeur d'un <tt>Adn</tt>. <p>
    *
    * Les lettres contenues dans la chaine <tt>cas</tt>
    * vont etre utilisees pour generer un <tt>Adn</tt>. <p>
    *
    * @param cas le cas etudie (PIN, ou PINM)
    */
  public Adn(String cas) 
  {
    this.cas = new String(cas);
    adn = genererCles();
  }
  
  /**
    * Affiche le genotype dans l'ordre des gnes
    */
  public void affiche()
  {
    int i;
    StringBuffer geno = new StringBuffer();
    if(adn.size() > 0)
      geno.append(getValeurGene(1));
    for(i = 2;i <= adn.size();i++)
    {
      geno.append(", [");
      geno.append(i);
      geno.append("] ");
      geno.append(getValeurGene(i));
    }
    System.out.println(String.valueOf(geno));
  }
  
  /**
    * Methode qui retourne  le cas etudier. <p>
    *
    * @return le cas etudie.
    */
  public final String getCas()
  {
    return new String(cas);
  }
  
  /**
    * Retourne le nombre de genes contenus dans l'<tt>Adn</tt>. <p>
    *
    * @return le nombre de genes contenus dans l'<tt>Adn</tt>.
    */
  public final int getSize()
  {
    if(adn == null)
      return 0;
    return adn.size();
  }
  
  /**
    * Constructeur sans parametre de <tt>Adn</tt>. <p>
    *
    * Ce constructeur ne <b><u>doit rien faire</u></b>, il sert uniquement dans
    * la methode <tt>clone()</tt>. <p>
    */
  protected Adn() 
  {
    ;
  }
  
  /**
    * Methode qui retourne le deplacement optimum en fonction du gene. <p>
    *
    * Un <tt>Adn</tt> parfait :
    * <ul>
    *   <li>le <tt>Ver</tt> doit se diriger ver la <tt>Nourriture</tt> la plus energetique.
    *   <li>il doit favoriser le deplacement Face, en cas d'egualite.
    * </ul>
    *
    * Exemple d'un <tt>Adn</tt> parfait :<br>
    * Si on prend 2 types de nourritures : ('N',100) et ('P',0).<br>
    * Alors
    * <ul>
    *   <li>"III" => "FGD"
    *   <li>"NII" => "GFD"
    *   <li>"INI" => "GFD"
    *   <li>"NNI" => "FGD"
    *   <li>"PNP" => "FGD"
    *   <li>"PII" => "GFD"
    *   <li>...
    * </ul>
    *
    * <b>ATTENTION !</b>
    * <ul>
    *   <li>Il est suppose ici que toutes les nourritures sont meilleures que le VIDE.
    *   <li>la lettre 'I' sera concideree comme inmangeable et pas une nourriture.
    *   <li>Les Murs n'existent pas, et ne sont pas pris en compte !
    * </ul>
    *
    * @param gene le gene codant la vision du ver (exp : "INP")
    * @param tn la table des <tt>Nourriture</tt>
    *
    * @return le deplacement optimal (exp pour "INP" => si N > P, on obtient
    * "FDG")
    */
  protected String rendreGeneParfait(String gene, TableNourriture tn)
  {
    
    // Il me faut le tableau des nourritures triees de la plus energetique
    
    // a la moins energetique
    Nourriture tab[] = tn.lister(true);
    int i, j, k;
    
    // cas particulier
    if((gene.charAt(0) == 'I') && (gene.charAt(1) == 'I') && (gene.charAt(2) == 'I'))
      return "FGD";
    
    // Une seule nourriture
    for(i = 0;i < tab.length;i++)
      if((gene.charAt(0) == tab[i].getType()) && (gene.charAt(1) == 'I') && (gene.charAt(2) == 'I'))
        return "GFD";
    for(i = 0;i < tab.length;i++)
      if((gene.charAt(0) == 'I') && (gene.charAt(1) == tab[i].getType()) && (gene.charAt(2) == 'I'))
        return "FGD";
    for(i = 0;i < tab.length;i++)
      if((gene.charAt(0) == 'I') && (gene.charAt(1) == 'I') && (gene.charAt(2) == tab[i].getType()))
        return "DFG";
    
    // 2 nourritures avec I
    
    // On rappelle que le tab contient les nourritures classees par ordre decroissant de valeurs
    
    // energetiques, c.a.d que si i<j tab[i].getValeur()>tab[j].getValeur().
    for(i = 0;i < tab.length;i++)
      for(j = 0;j < tab.length;j++)
        if((gene.charAt(0) == tab[i].getType()) && (gene.charAt(1) == tab[j].getType()) && (gene.charAt(2) == 'I'))
        {
          if(i < j)
            return "GFD";
          else
            return "FGD";
        }
    for(i = 0;i < tab.length;i++)
      for(j = 0;j < tab.length;j++)
        if((gene.charAt(0) == tab[i].getType()) && (gene.charAt(1) == 'I') && (gene.charAt(2) == tab[j].getType()))
        {
          if(i < j)
            return "GDF";
          else
            return "DGF";
        }
    for(i = 0;i < tab.length;i++)
      for(j = 0;j < tab.length;j++)
        if((gene.charAt(0) == 'I') && (gene.charAt(1) == tab[i].getType()) && (gene.charAt(2) == tab[j].getType()))
        {
          if(i < j)
            return "FDG";
          else
            return "FDG";
        }
    
    // 3 Nourritures
    
    // On rappelle que le tab contient les nourritures classees par odre decroissant de valeurs
    
    // energetiques, c.a.d que si i<j tab[i].getValeur()>tab[j].getValeur().
    for(i = 0;i < tab.length;i++)
      for(j = 0;j < tab.length;j++)
        for(k = 0;k < tab.length;k++)
          if((gene.charAt(0) == tab[i].getType()) && (gene.charAt(1) == tab[j].getType()) && (gene.charAt(2) == tab[k].getType()))
          {
            
            // Cas ou les trois nourritures sont les memes
            if((i == j) && (j == k))
              return "FGD";
            
            // Cas ou deux des trois nourritures sont les memes
            if((i == j) && (j < k))
              return "FGD";
            if((i == k) && (j < k))
              return "FGD";
            if((j == k) && (i < k))
              return "GFD";
            if((j == k) && (i > k))
              return "FDG";
            if(min(i, j, k) == i)
            {
              if(min(j, k) == j)
                return "GFD";
              else
                return "GDF";
            }
            else
              if(min(i, j, k) == j)
              {
                if(min(i, k) == i)
                  return "FGD";
                else
                  return "FDG";
              }
              else
                if(min(i, j, k) == k)
                {
                  if(min(i, j) == i)
                    return "DGF";
                  else
                    return "DFG";
                }
          }
    return "???";
  }
  
  /**
    * Permet de generer toutes les cles en fonction du
    * String passe en parametre. <P>
    *
    * @return une table representant l'<tt>Adn</tt>
    */
  protected Hashtable genererCles()
  {
    int i;
    Hashtable resu = new Hashtable();
    int nbCas = cas.length();
    for(i = 1;i <= Math.pow(nbCas, 3);i++)
    {
      String s = convert(i);
      resu.put(s, dirRandom());
    }
    return resu;
  }
  
  /**
    * Renvoie la chaine correspondant a une combinaison des  
    * differents cas que peut rencontrer le ver (les nourritures,
    * les murs, immangeable...) ; cette fonction est utilisee 
    * pour generer toutes les cles. <P>
    *
    * @param nb entier pour indexer une situation face a laquelle se 
    * trouve le ver
    *
    * @return une chaine representant le gene <i>"III"</i> par exemple
    */
  protected final String convert(int nb)
  {
    int i;
    int nbCas = cas.length();
    StringBuffer result = new StringBuffer();
    for(i = 1;i <= 3;i++)
    {
      result.append(cas.charAt(nb % nbCas));
      nb = nb / nbCas;
    }
    return String.valueOf(result);
  }
  
  /**
    * Permet de choisir un deplacement au hasard. <P>
    *
    * @return le deplacement sous forme de chaine.
    */
  protected static String dirRandom()
  {
    switch(Generateur.tirerIntEntre(6))
    {
      case 0:
        return "FGD";
      
      case 1:
        return "FDG";
      
      case 2:
        return "DFG";
      
      case 3:
        return "DGF";
      
      case 4:
        return "GFD";
      
      default:
        return "GDF";
    }
  }
  
  /**
    * Methode qui retourne le minimum de trois nombres. <p>
    *
    * @param a un nombre
    * @param b un nombre
    * @param c un nombre
    *
    * @ return le min(a,b,c)
    */
  private final static int min(int a, int b, int c)
  {
    if(a == b)
      return min(a, c);
    if(a == c)
      return min(a, b);
    if(b == c)
      return min(b, a);
    if((a == b) && (a == c))
      return a;
    if((a < b) && (b < c))
      return a;
    if((b < a) && (a < c))
      return b;
    if((c < a) && (a < b))
      return c;
    return c;
  }
  
  /**
    * Methode qui retourne le minimum de deux nombres. <p>
    *
    * @param a un nombre
    * @param b un nombre
    *
    * @ return le min(a,b)
    */
  private final static int min(int a, int b)
  {
    if(a < b)
      return a;
    return b;
  }
  
  /** Indique si l'<tt>Adn</tt> est parfait ou non. <p> */
  protected boolean estParfait = false;
  
  /** Le cas etudie. <p> */
  protected String cas = null;
  
  /** Le genes sont ranges dans une <tt>Hashtable</tt>. <p> */
  protected Hashtable adn = null;
}
